(function () { 'use strict';

    /**
     * Author: Jason Farrell
     * Author URI: http://useallfive.com/
     *
     * Description: Checks if a DOM element is truly visible.
     * Package URL: https://github.com/UseAllFive/true-visibility
     *
     * ----------------------------------------------------------
     *
     * This is a fork that uses ES6 export and doesn't extend the
     * native Elenent object.
     *
     * Forked by: Dmirty Sobolev
     * 
     * Fork URL: https://github.com/dsblv/true-visibility
     *
     */
    var isVisible = function() {

        'use strict';

        /**
         * Checks if a DOM element is visible. Takes into
         * consideration its parents and overflow.
         *
         * @param (el)      the DOM element to check if is visible
         *
         * These params are optional that are sent in recursively,
         * you typically won't use these:
         *
         * @param (t)       Top corner position number
         * @param (r)       Right corner position number
         * @param (b)       Bottom corner position number
         * @param (l)       Left corner position number
         * @param (w)       Element width number
         * @param (h)       Element height number
         */
        function _isVisible(el, t, r, b, l, w, h) {
            var p = el.parentNode,
                    VISIBLE_PADDING = 2;

            if ( !_elementInDocument(el) ) {
                return false;
            }

            //-- Return true for document node
            if ( 9 === p.nodeType ) {
                return true;
            }

            //-- Return false if our element is invisible
            if (
                 '0' === _getStyle(el, 'opacity') ||
                 'none' === _getStyle(el, 'display') ||
                 'hidden' === _getStyle(el, 'visibility') ||
                 ('hidden' === _getStyle(el, 'overflow-x') && ('0px' == _getStyle(el, 'width') || '0px' == _getStyle(el, 'height')))
            ) {
                return false;
            }

            if (
                'undefined' === typeof(t) ||
                'undefined' === typeof(r) ||
                'undefined' === typeof(b) ||
                'undefined' === typeof(l) ||
                'undefined' === typeof(w) ||
                'undefined' === typeof(h)
            ) {
                t = el.offsetTop;
                l = el.offsetLeft;
                b = t + el.offsetHeight;
                r = l + el.offsetWidth;
                w = el.offsetWidth;
                h = el.offsetHeight;
            }
            //-- If we have a parent, let's continue:
            if ( p ) {
                //-- Check if the parent can hide its children.
                if ( ('hidden' === _getStyle(p, 'overflow') || 'scroll' === _getStyle(p, 'overflow')) ) {
                    //-- Only check if the offset is different for the parent
                    if (
                        //-- If the target element is to the right of the parent elm
                        l + VISIBLE_PADDING > p.offsetWidth + p.scrollLeft ||
                        //-- If the target element is to the left of the parent elm
                        l + w - VISIBLE_PADDING < p.scrollLeft ||
                        //-- If the target element is under the parent elm
                        t + VISIBLE_PADDING > p.offsetHeight + p.scrollTop ||
                        //-- If the target element is above the parent elm
                        t + h - VISIBLE_PADDING < p.scrollTop
                    ) {
                        //-- Our target element is out of bounds:
                        return false;
                    }
                }
                //-- Add the offset parent's left/top coords to our element's offset:
                if ( el.offsetParent === p ) {
                    l += p.offsetLeft;
                    t += p.offsetTop;
                }
                //-- Let's recursively check upwards:
                return _isVisible(p, t, r, b, l, w, h);
            }
            return true;
        }

        //-- Cross browser method to get style properties:
        function _getStyle(el, property) {
            if ( window.getComputedStyle ) {
                return document.defaultView.getComputedStyle(el,null)[property];
            }
            if ( el.currentStyle ) {
                return el.currentStyle[property];
            }
        }

        function _elementInDocument(element) {

            if (!(element instanceof Element))
                return false;

            while (element = element.parentNode) {
                if (element == document) {
                        return true;
                }
            }
            return false;
        }

        return _isVisible(this);

    };

    function extend (methods) {
        for (var method in methods)
            if (methods.hasOwnProperty(method))
                this.prototype[method] = methods[method];

        return this;
    }

    function extendOwn (properties) {

        for (var property in properties) {
            this[property] = properties[property];
        }

        return this;
    }

    var Finding = function (position, searcher, haystack) {
        this.position = position;
        this.active   = false;
        this.searcher = searcher;
        this.haystack = haystack || searcher.haystack;

        var node           = document.createElement('span');
            node.className = searcher.settings.className;

        extendOwn.call(node.style, searcher.settings.findingStyle);

        this.node = node;
    }

    Finding.extend = extend;

    Finding.extend({
        appear: function (callback) {
            var text      = this.searcher.needle,
                haystack  = this.haystack,
                node      = this.node;

            var after           = (this.position > 0) ? haystack.splitText(this.position) : haystack;
            
                text            = after.nodeValue.substring(0, text.length);
                after.nodeValue = after.nodeValue.substring(text.length);

            node.appendChild(document.createTextNode(text));

            haystack.parentNode.insertBefore(node, after);

            callback(after);

            return this;
        },

        hilight: function () {
            this.active = true;

            extendOwn.call(this.node.style, this.searcher.settings.hilightStyle);

            this.node.scrollIntoView();

            var top = this.node.getBoundingClientRect().top;

            if (top < 30)
                window.scrollBy(0, -document.documentElement.clientHeight/2);

            return this;
        },
        extinguish: function () {
            this.active = false;

            extendOwn.call(this.node.style, this.searcher.settings.findingStyle);

            return this;
        },

        disappear: function () {

            var node   = this.node,
                parent = node.parentNode;

            parent.replaceChild(node.firstChild, node);
            parent.normalize();

            return this;
        }
    });

    var Searcher = (function (settings) {
        return {
            findings : [],
            settings : settings,
            search   : function (needle) {
                this.fallback()
                    ._searchFor(needle);

                _Searcher.setInfo(this.current(), this.count());

                return this;

            },

            _searchFor: function (needle, haystack) {
                var haystack    = haystack || this.settings.haystack,
                    searcher    = this;

                    this.needle = needle;

                /* no one searches for 1-letter words */
                if (needle.length < 1)
                    return;


                /* we don't search inside hilighted spans, scripts & forms */
                if (haystack.className == this.settings.className ||
                    !isVisible.call(haystack.parentNode)          ||
                    typeof haystack.tagName != 'undefined'        &&
                           haystack.tagName.match(/(script|input|button|textarea|select)/i))
                    return;

                /* we don't search inside complex elements: instead we go inside its children */
                if (typeof haystack.hasChildNodes != 'undefined' && haystack.hasChildNodes())
                    for (var i = 0; i < haystack.childNodes.length; i++) {
                        this._searchFor(needle, haystack.childNodes[i]);
                    }


                /* if element is text_node, we look closer */
                if (haystack.nodeType === 3) {

                    var match = haystack.nodeValue.toLowerCase().indexOf(needle.toLowerCase());

                    /* collect all matches */
                    if (match > -1) {
                        var finding = new Finding(match, this, haystack);

                        /* when match is found, we make it appear (as finding) */
                        finding.appear(function (haystack) {
                            /* Finding.appear() accepts callback in which we sort findings & search in newly-created areas */
                            searcher._searchFor(needle, haystack);
                        });

                        this.findings.push(finding);

                    }  
                }

                return this._resort();
            },

            _resort: function () {
                /* sorting findings based on their screen position */
                this.findings.sort(function (a, b) {
                    var a = a.node.getBoundingClientRect(),
                        b = b.node.getBoundingClientRect();

                    if (a.top + window.scrollY < b.top + window.scrollY)
                        return -1;

                    if (a.top + window.scrollY == b.top + window.scrollY)
                        if ((a.left + window.scrollX - b.left + window.scrollX) < 0)
                            return -1;

                    return 1;
                });

                if (this.findings.length)
                        this.hilight(0);

                return this;
            },

            hilight: function (index) {
                /* extinguish all to hilight one */
                this.hilighted = index || 0;

                for (var i = 0; i < this.findings.length; i++)
                    this.findings[i].extinguish();
                    

                if (this.hilighted > -1)
                    this.findings[this.hilighted].hilight();

                return this;

            },

            next: function () {

                if (++this.hilighted == this.findings.length)
                    this.hilighted = 0;

                _Searcher.setInfo(this.current(), this.count());

                return this.hilight(this.hilighted);

            },

            previous: function () {

                if (--this.hilighted < 0)
                    this.hilighted = this.findings.length - 1;

                _Searcher.setInfo(this.current(), this.count());

                return this.hilight(this.hilighted);

            },

            fallback: function () {
                /* unhilight everything */

                for (var i = 0; i < this.findings.length; i++)
                    this.findings[i].disappear();

                this.needle    = '';
                this.findings  = [];
                this.hilighted = -1;

                return this;
            },

            count: function () {
                return this.findings.length;
            },

            current: function () {
                return this.hilighted;
            }
        }

    })({
        needle       : '',
        haystack     : document.body,
        className    : 'orbitum-search-finding',
        findingStyle : {
            backgroundColor : 'rgb(255, 255, 0)'
        },
        hilightStyle : {
            backgroundColor : 'rgb(255, 150, 50)'
        }
    }); 


    //export default Searcher;

    window.Searcher = Searcher;

})();