'use strict';

// Logging

function consoleLog(s) {
    if (AQM_DEBUG_LOG_ENABLED) {
        console.log(s)
    }
}

function consoleLogVerbose(s) {
    if (AQM_DEBUG_LOG_ENABLED && AQM_DEBUG_LOG_VERBOSE) {
        console.log(s)
    }
}

// Utilities

function getComputedStyleProperty(el, propName) {
    if (window.getComputedStyle) {
        return window.getComputedStyle(el, null)[propName];
    } else if (el.currentStyle) {
        return el.currentStyle[propName];
    }
}

function getRangeAncestorContainer(range) {
    var ancestorEl = range.commonAncestorContainer
    if (ancestorEl && ancestorEl.nodeType == Node.TEXT_NODE) {
        // text -> actual node
        ancestorEl = ancestorEl.parentNode
    }
    return ancestorEl
}

function findEnclosingBlock(el, tagName) {
    var tagNameUpper = tagName.toUpperCase()
    while (true) {
        if (el) {
            if (el.nodeType == Node.ELEMENT_NODE && el.tagName == tagNameUpper) {
                return el
            }
            el = el.parentNode
        } else {
            return null
        }
    }
}

function safeDecodeSnippet(s) {
    if (s != null) {
        return decodeURIComponent(s.replace(/\+/g, '%20'))
    }
    return null
}

function escapeHtmlString(s) {
    return s.replace(/[&<"']/g, function(m) {
        switch (m) {
            case '&':
                return '&amp;'
            case '<':
                return '&lt;'
            case '"':
                return '&quot;'
            case '\'':
                return '&#039;'
            default:
                return m
        }
    })
}

function limitStringTo(s, maxLength) {
    if (s && s.length > maxLength) {
        return s.substring(0, maxLength)
    }
    return s
}

function getRangeSelectedText(range) {
    var startContainer = range.startContainer
    var endContainer = range.endContainer
    var selectedText

    if (startContainer && startContainer.nodeType == Node.TEXT_NODE) {
        // Get selected text
        if (startContainer === endContainer) {
            // Easy, same node
            consoleLog('Same start / end text node ' + startContainer)
            selectedText = startContainer.textContent.substring(range.startOffset, range.endOffset)
        } else if (endContainer && startContainer.parentNode === endContainer.parentNode) {
            // Split text nodes
            consoleLog('Diff start / end text nodes with same parent ' + startContainer.parentNode)

            var currText = startContainer.textContent.substring(range.startOffset)
            var currNode = startContainer.nextSibling
            while (currNode && currNode != endContainer) {
                if (currNode.nodeType == Node.TEXT_NODE) {
                    currText = currText + currNode.textContent
                } else if (currNode.nodeType == Node.ELEMENT_NODE) {
                    currText = currText + currNode.innerText
                }
                if (currText && currText.length > AQM_MAX_SELECTION_TEXT_LEN) {
                    break
                }
                currNode = currNode.nextSibling
            }
            if (currNode && currNode == endContainer) {
                if (currNode.nodeType == Node.TEXT_NODE) {
                    currText = currText + currNode.textContent.substring(0, range.endOffset)
                }
                selectedText = currText
            }
        }
    }

    selectedText = limitStringTo(selectedText, AQM_MAX_SELECTION_TEXT_LEN)

    return selectedText
}

function domRectToString(rect) {
    if (rect === null) {
        return 'null'
    } else if (rect == undefined) {
        return 'undefined'
    } else {
        return '[left: ' + rect.left + ', top: ' + rect.top + ', right: ' + rect.right +
            ', bottom: ' + rect.bottom + ']'
    }
}

function createTagNameMap() {
    var map = {}

    for (var i = 0; i < arguments.length; i++) {
        var arg = arguments[i].toUpperCase()
        map[arg] = 1
    }

    return map
}

function copyNodeList(list) {
    var ar = []
    for (var i = 0; i < list.length; ++ i) {
        ar.push(list[i])
    }
    return ar
}

function equalsIgnoreCase(s1, s2) {
//    return s1.localeCompare(s2, 'en', { sensitivity: 'base' }) == 0
    return s1.toUpperCase() == s2.toUpperCase()
}

function nodeHasTagNameIgnoreCase(node, value) {
    var there = node.tagName
    if (there) {
        return equalsIgnoreCase(there, value)
    }
    return false
}

function nodeHasAttributeValueIgnoreCase(node, attr, value) {
    var there = node.getAttribute(attr)
    if (there) {
        return equalsIgnoreCase(there, value)
    }
    return false
}

function areAllTextChildrenWhitespace(node) {
    for (var sub = node.firstChild; sub != null; sub = sub.nextSibling) {
        if (sub.nodeType == Node.TEXT_NODE) {
            var value = sub.nodeValue
            if (value && value.trim() != '') {
                return false
            }
        }
    }
    return true
}

function onFormSubmit() {
    return false
}

function disableForms() {
    var forms = document.getElementsByTagName('form');
    for (var i = 0; i < forms.length; ++i) {
        if (forms[i].method.toUpperCase() === 'POST') {
            forms[i].onsubmit = onFormSubmit()
            elements = forms[i].elements;
            for (var j = 0; j < elements.length; ++j) {
                if (elements[j].type != 'submit') {
                    elements[j].disabled = true;
                }
            }
        }
    }
}

// New Chrome version?

function isNewChrome(needVersion) {
    var userAgent = navigator.userAgent
    if (userAgent) {
        consoleLog("isNewChrome: " + userAgent)
        var raw = navigator.userAgent.match(/Chrom[a-z]+\/([0-9]+)\./);
        if (raw && raw[1]) {
            var version = parseInt(raw[1])
            consoleLog("isNewChrome: " + version + ", need " + needVersion)
            return version >= needVersion
        }
    }
    return false
}

// We actually use WebView 60 and newer, the built-in are 37 (Android 5.0), 39 (Android 5.1) or 44 (Android 6.0)

var AQM_IS_NEW_CHROME_57 = isNewChrome(57)

function checkPasteNothingBug() {
    if (!AQM_IS_NEW_CHROME_57) {
        AQM_EDIT_CALLBACK.showPasteNothingBug()
    }
}

// Compatibility utilities

function setStyleCompat(node, style) {
    if (AQM_IS_NEW_CHROME_57) {
        try {
            node.style = style
        } catch (e) {
            consoleLog('setStyle: error ' + e + ' will try setAttribute instead: ' + style)
            node.setAttribute('style', style)
        }
    } else {
        consoleLog('setStyle: not new chrome, will use setAttribute instead: ' + style)
        node.setAttribute('style', style)
    }
}

function querySelectorCompat(node, sel1, sel2) {
    if (AQM_IS_NEW_CHROME_57) {
        try {
            return node.querySelector(sel1)
        } catch (e) {
            consoleLog('querySelector: error ' + e + ' on \"' + sel1 + '\" will try \"' + sel2 + '\" instead')
            return node.querySelector(sel2)
        }
    } else {
        consoleLog('querySelector: not new chrome, will use \"' + sel2 + '\" instead')
        return node.querySelector(sel2)
    }
}

// Our editable area

var AQM_EDIT_AREA = null

// Original node when replying

var AQM_ORIGINAL = null

// Dirty (not saved) tracking

var AQM_IS_DIRTY_POSTED = false

function doIsDirty() {
    AQM_IS_DIRTY_POSTED = false
    AQM_EDIT_CALLBACK.editContentIsDirty()
}

function postIsDirty() {
    if (!AQM_IS_DIRTY_POSTED) {
        AQM_IS_DIRTY_POSTED = true
        window.setTimeout(doIsDirty, 0)
    }
}

// Images

var MAX_IMAGE_WIDTH = 300
var MAX_IMAGE_HEIGHT = 240

// Selection state

var AQM_SELECTION_CHECK_POSTED = false

var AQM_STATE_BOLD = 0x0001
var AQM_STATE_ITALIC = 0x0002
var AQM_STATE_UNDERLINED = 0x004
var AQM_STATE_STRIKETHROUGH = 0x0008

var AQM_STATE_ALIGN_LEFT = 0x0100
var AQM_STATE_ALIGN_RIGHT = 0x0200
var AQM_STATE_ALIGN_JUSTIFY = 0x0400
var AQM_STATE_ALIGN_CENTER = 0x0800

var AQM_STATE_URL_LINK = 0x1000

var AQM_STATE_BLOCKQUOTE = 0x10000
var AQM_STATE_UNORDERED_LIST = 0x20000

var AQM_STATE_ARRAY = [
    // Basic
    'bold', AQM_STATE_BOLD,
    'italic', AQM_STATE_ITALIC,
    'underline', AQM_STATE_UNDERLINED,
    'strikethrough', AQM_STATE_STRIKETHROUGH,
    // Alignment
    'justifyLeft', AQM_STATE_ALIGN_LEFT,
    'justifyRight', AQM_STATE_ALIGN_RIGHT,
    'justifyFull', AQM_STATE_ALIGN_JUSTIFY,
    'justifyCenter', AQM_STATE_ALIGN_CENTER,
    // Block
    'blockquote', AQM_STATE_BLOCKQUOTE,
    'insertUnorderedList', AQM_STATE_UNORDERED_LIST
]

var AQM_BLOCK_TAGS_MAP = createTagNameMap(
    'div', 'blockquote', 'table', 'tr',
    'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
    'pre', 'p', 'ol', 'ul', 'li'
)

var AQM_MAX_SELECTION_TEXT_LEN = 500

function isEmptyDivSelection(range) {
    if (range.startContainer == range.endContainer &&
        range.startOffset == 0 && range.endOffset == 0) {
        var startContainer = range.startContainer
        if (startContainer && startContainer.nodeType == Node.ELEMENT_NODE && startContainer.tagName == 'DIV') {
            var firstElementChild = startContainer.firstElementChild
            if (firstElementChild && firstElementChild.tagName == 'BR' && startContainer.childElementCount == 1) {
                if (areAllTextChildrenWhitespace(startContainer)) {
                    return true
                }
            }
        }
    }

    return false
}

// Not all WebView / Chrome versions have DOMRect (!!!)
function AqmSimpleRect(l, t, w, h) {
    this.left = l
    this.top = t
    this.right = l + w
    this.bottom = t + h
    this.width = w
    this.height = h
}

function doSelectionCheck() {
    AQM_SELECTION_CHECK_POSTED = false
    consoleLog('doSelectionCheck')

    var effects = 0

    var ancestorEl
    var boundingRect = null
    var selectedText = null
    var selectedLink = null
    var selection = document.getSelection()
    var selectionIsCollapsed = false

    if (selection) {
        selectionIsCollapsed = selection.isCollapsed
        if (selection.rangeCount > 0) {
            var range = selection.getRangeAt(0)
            if (range) {
                var startContainer = range.startContainer
                var endContainer = range.endContainer

                // Scrolling
                var scrollY = window.scrollY
                if (scrollY != 0) {
                    // AQM-1246, WebView wants to scroll, sometimes, and we don't want that
                    consoleLog('window.scrollY = ' + scrollY)
                    window.scrollBy(0, -scrollY)
                }

                // Ancestor
                ancestorEl = getRangeAncestorContainer(range)

                // Selected text
                if (!range.collapsed) {
                    selectedText = getRangeSelectedText(range)
                }

                // Check if there is an anchor above
                if (startContainer && startContainer.nodeType == Node.TEXT_NODE) {
                    if (startContainer.parentNode && startContainer.parentNode.nodeType == Node.ELEMENT_NODE) {
                        if (startContainer.parentNode.tagName == 'A') {
                            effects |= AQM_STATE_URL_LINK
                            selectedLink = startContainer.parentNode.getAttribute('href')
                        }
                    }
                }

                if (selectionIsCollapsed && startContainer && isEmptyDivSelection(range)) {
                    // Special case: WebView sometimes returns wrong client rect for empty lines (<div><br></div>)
                    // Use the empty div's bounding rect, but it's wrong too! Calculate from offsets
                    var offsetTop = 0
                    for (var cur = startContainer; cur; cur = cur.offsetParent) {
                        offsetTop += cur.offsetTop
                    }

                    consoleLog('selection is an empty DIV, offsetTop = ' + offsetTop)
                    boundingRect = startContainer.getBoundingClientRect()

                    if (boundingRect != null && boundingRect.top != offsetTop) {
                        consoleLog('selection adjustment ' + boundingRect.top + ' to offsetTop = ' + offsetTop)
                        boundingRect = new AqmSimpleRect(boundingRect.left, offsetTop, boundingRect.width,
                            boundingRect.height)
                    }
                } else {
                    // Now we want the currently tracked selection "handle"
                    if (selection.focusNode) {
                        var r = document.createRange()
                        r.setStart(selection.focusNode, selection.focusOffset)
                        r.setEnd(selection.focusNode, selection.focusOffset)
                        range = r
                    }

                    // Get bounding rect
                    boundingRect = range.getBoundingClientRect()
                    if (selectionIsCollapsed && boundingRect && boundingRect.height == 0 &&
                            AQM_EDIT_AREA.childElementCount == 0) {
                        // Special case when there is nothing in the editor, at all
                        boundingRect = AQM_EDIT_AREA.getBoundingClientRect()
                        if (boundingRect != null) {
                            boundingRect = new AqmSimpleRect(boundingRect.left, boundingRect.top,
                                boundingRect.width, 19)
                        }
                    } else if (!boundingRect || boundingRect.height == 0) {
                        // Did not get anything
                        if (startContainer && startContainer.nodeType == Node.ELEMENT_NODE &&
                                startContainer === range.endContainer) {
                            // If selection is collapsed, use the container
                            boundingRect = startContainer.getBoundingClientRect()
                        } else if (selection.focusNode) {
                            // Try to use the focus node itself
                            var parent
                            if (selection.focusNode.nodeType == Node.ELEMENT_NODE) {
                                parent = selection.focusNode.childNodes[selection.focusOffset]
                            }
                            if (!parent) {
                                parent = selection.focusNode
                            }

                            while (parent && parent.nodeType != Node.ELEMENT_NODE) {
                                parent = parent.parentNode
                            }
                            if (parent) {
                                boundingRect = parent.getBoundingClientRect()
                            }
                        }
                    }
                }

                consoleLog('selection range = ' + range.startOffset + ":" + range.endOffset + ", ancestor = " +
                    ancestorEl + ", text = " + selectedText + ", link = " +
                    selectedLink +", rect = " + domRectToString(boundingRect))
            }
        }
    }

    // Current state of "boolean" effects
    for (var i = 0; i < AQM_STATE_ARRAY.length; i += 2) {
        if (document.queryCommandState(AQM_STATE_ARRAY[i])) {
            effects |= AQM_STATE_ARRAY[i + 1]
        }
    }

    // Non-boolean effects
    var foreColor = document.queryCommandValue('forecolor')
    var backColor = document.queryCommandValue('backcolor')
    var fontName

    if (ancestorEl) {
        fontName = getComputedStyleProperty(ancestorEl, 'font-family')
    }

    consoleLog('Fore color = ' + foreColor + ", back color = " + backColor + ", font name = " + fontName)

    // If we could not get bounding rect, assume empty
    if (boundingRect == null) {
        boundingRect = new AqmSimpleRect(0, 0, 0, 0)
    }

    AQM_EDIT_CALLBACK.updateSelectionState(effects, foreColor, backColor, fontName,
        selectedText, selectedLink,
        boundingRect.left, boundingRect.top, boundingRect.right, boundingRect.bottom,
        selectionIsCollapsed)
}

function selectSelectedAnchor() {
    var selection = document.getSelection()
    if (selection) {
        if (selection.rangeCount > 0) {
            var range = selection.getRangeAt(0)
            if (range) {
                var startContainer =  range.startContainer
                var isSingleNode = startContainer && startContainer === range.endContainer

                if (isSingleNode && startContainer.nodeType == Node.TEXT_NODE) {
                    if (startContainer.parentNode && startContainer.parentNode.nodeType == Node.ELEMENT_NODE) {
                        if (startContainer.parentNode.tagName == 'A') {
                            var nodeA = startContainer.parentNode
                            var rangeA = document.createRange()

                            consoleLog('Selecting anchor node: ' + nodeA)

                            rangeA.selectNode(nodeA)
                            selection.removeAllRanges()
                            selection.addRange(rangeA)
                        }
                    }
                }
            }
        }
    }
}

function ensureSelectBeforeInsert() {
    var selection = document.getSelection()
    if (selection && selection.rangeCount <= 0) {
        var range = document.createRange()

        var node = document.getElementById('aqm-greeting')
        if (node) {
            range.setStartAfter(node)
            range.collapse(true)
        } else if ((node = document.getElementById('aqm-toggle')) != null ||
            (node = document.getElementById('aqm-original')) != null ||
            (node = document.getElementById('aqm-signature')) != null) {
            range.setStartBefore(node)
            range.collapse(true)
        } else {
            range.selectNodeContents(AQM_EDIT_AREA)
            range.collapse(false)
        }

        selection.removeAllRanges()
        selection.addRange(range)
    }
}

function postSelectionCheck() {
    if (!AQM_SELECTION_CHECK_POSTED) {
        AQM_SELECTION_CHECK_POSTED = true;
        window.setTimeout(doSelectionCheck, 0)
    }
}

// Special processing for enter key at the beginning / end of some nodes

var AQM_SPECIAL_NAMES_PREFIX = 'aqm-'

var AQM_SPECIAL_NAMES_ALL = [
    'aqm-greeting', 'aqm-signature', 'aqm-toggle', 'aqm-original' ]

var AQM_SPECIAL_NAMES_FIX_AFTER_END = [
    'aqm-greeting', 'aqm-signature', 'aqm-toggle', 'aqm-original' ]

var AQM_SPECIAL_NAMES_FIX_BEFORE_START = [
    'aqm-greeting', 'aqm-signature', 'aqm-toggle', 'aqm-original' ]

function isTextNode(node) {
	return node.nodeType == Node.TEXT_NODE
}

function isTextNodeEmpty(node) {
    var text = node.nodeValue
    var trim = text.replace(/[ \u00A0\n]+$/, '')
    return trim == ''
}

function isTextNodeCanSkip(node) {
    return isTextNode(node) && isTextNodeEmpty(node)
}

function isTextNodeEnd(s, end) {
    var text = s.substring(end)
    var trim = text.replace(/[ \u00A0\n]+$/, '')
    return trim == ''
}

function isTextNodeStart(s, start) {
    var text = s.substring(0, start)
    var trim = text.replace(/^[ \u00A0\n]+/, '')
    return trim == ''
}

function isCommentNode(node) {
    return node.nodeType == Node.COMMENT_NODE
}

function isDivNode(node) {
    return node.nodeType == Node.ELEMENT_NODE && node.tagName == 'DIV'
}

function isBrNode(node) {
    return node.nodeType == Node.ELEMENT_NODE && node.tagName == 'BR'
}

function isSpanNode(node) {
    return node.nodeType == Node.ELEMENT_NODE && node.tagName == 'SPAN'
}

function isDivNodeEmpty(node) {
    if (node.childElementCount == 0) {
        var list = node.childNodes
        if (!list || list.length == 0) {
            return true
        }
        if (list.length == 1) {
            var child = list[0]
            if (child.nodeType == Node.TEXT_NODE && isTextNodeEmpty(child)) {
                return true
            }
        }
    }
    return false
}

function isDivNodeLineBreak(node) {
    var list = node.childNodes
    if (list && list.length >= 1) {
        var seenBreak = false
        for (var i = 0; i < list.length; ++ i) {
            var child = list[0]
            if (child.nodeType == Node.ELEMENT_NODE && child.tagName == 'BR') {
                if (seenBreak) {
                    return false
                }
                seenBreak = true
            } else if (!isTextNodeCanSkip(child)) {
                return false
            }
        }
        return seenBreak
    }
    return false
}

function isAllowedWhiteSpaceNode(node) {
    var nodeType = node.nodeType
    if (nodeType == Node.ELEMENT_NODE) {
        if (node.tagName == 'BR') {
            // Allowed
            return true
        } else if (node.tagName == 'DIV') {
            // Allowed
            return isDivNodeEmpty(node)
        }
    } else if (nodeType == Node.TEXT_NODE) {
        // Allowed
        return isTextNodeEmpty(node)
    } else if (nodeType == Node.COMMENT_NODE) {
        // Allowed
        return true
    }
    return false
}

function isLastNodeInBlock(node) {
    if (node) {
        node = node.nextSibling

        while (true) {
            if (node == null) {
                return true
            }

            if (isAllowedWhiteSpaceNode(node)) {
                // Allowed
                node = node.nextSibling
            } else {
                // Something we're not supposed to have
                break
            }
        }
    }

    return false
}

function isFirstNodeInBlock(node) {
    if (node) {
        node = node.previousSibling

        while (true) {
            if (node == null) {
                return true
            }

            if (isAllowedWhiteSpaceNode(node)) {
                // Allowed
                node = node.previousSibling
            } else {
                // Something we're not supposed to have
                break
            }
        }
    }

    return false
}

function isNodeIdSpecialInList(curr, idList) {
    if (curr.nodeType == Node.ELEMENT_NODE) {
        var id = curr.id
        return id && id.startsWith(AQM_SPECIAL_NAMES_PREFIX) && idList.indexOf(id) >= 0
    }
    return false
}

function isAtNodeWithId(id, range) {
    var start = range.startContainer, end = range.endContainer
    if (start && start.nodeType == Node.ELEMENT_NODE && start.id == id) {
        if (start == end && range.startOffset == 0 && range.endOffset == 0) {
            return true
        }
    }
}

function isAtNodeTextEnd(idList, range) {
    if (isAtNodeWithId('aqm-toggle', range)) {
        // AQM-1256, our logic gets confused by "::before"
        return null
    }

    var curr = range.endContainer
    if (curr) {
        if (curr.nodeType == Node.TEXT_NODE) {
            // Selection inside text
            if (!isTextNodeEnd(curr.nodeValue, range.endOffset)) {
                return null
            }
        } else if (curr.nodeType == Node.ELEMENT_NODE) {
            // Selection between two element nodes with no text in between
            var list = curr.childNodes
            var save = curr
            curr = null

            if (list) {
                if (range.endOffset >= list.length) {
                    // End is beyond last node
                    curr = save
                } else if (range.endOffset == 0) {
                    // End is before 0th node
                    if (isLastNodeInBlock(list[0])) {
                        curr = save
                    }
                } else if (isLastNodeInBlock(list[range.endOffset - 1])) {
                    // End is after this child
                    curr = save
                }
            }
        } else {
            return null
        }

        while (curr) {
            if (isNodeIdSpecialInList(curr, idList)) {
                consoleLog('isAtNodeTextEnd id = ' + curr.id + ' -> true')
                return curr
            }
            if (!isLastNodeInBlock(curr)) {
                break
            }
            curr = curr.parentNode
        }
    }

    return null
}

function isAtNodeTextStart(idList, range) {
    if (isAtNodeWithId('aqm-toggle', range)) {
        // AQM-1256, our logic gets confused by "::before"
        return null
    }

    var curr = range.startContainer
    if (curr) {
        if (curr == AQM_EDIT_AREA) {
            // Selection inside edit area
            var child = range.startContainer.childNodes[range.startOffset]
            if (child && isNodeIdSpecialInList(child, idList)) {
                return child
            }
        }

        if (curr.nodeType == Node.TEXT_NODE) {
            // Selection inside text
            if (!isTextNodeStart(curr.nodeValue, range.startOffset)) {
                return null
            }
        } else if (curr.nodeType == Node.ELEMENT_NODE) {
             // Selection between two element nodes with no text in between
            var list = curr.childNodes
            var save = curr
            curr = null

            if (list) {
                if (range.startOffset == 0) {
                    // Start is before 0th node
                    curr = save
                } else if (isFirstNodeInBlock(list[range.startOffset])) {
                    // Start is before this child
                    curr = save
                }
            }
        } else {
            return null
        }

        while (curr) {
            if (isNodeIdSpecialInList(curr, idList)) {
                consoleLog('isAtNodeTextStart id = ' + curr.id + ' -> true')
                return curr
            }
            if (!isFirstNodeInBlock(curr)) {
                break
            }
            curr = curr.parentNode
        }
    }

    return null
}

function shouldSplitOriginalAtSelection(selection, range) {
    var node = getRangeAncestorContainer(range)

    while (true) {
        if (node == null || node == AQM_EDIT_AREA) {
            break
        }
        if (node.id == 'aqm-original') {
            return true
        }
        if (node.nodeType == Node.ELEMENT_NODE && node.tagName == 'A') {
            break
        }
        node = node.parentNode
    }

    return false
}

function splitOriginalAtSelection(selection, range) {
    if (selection && range && shouldSplitOriginalAtSelection(selection, range)) {

        consoleLog('Special case for enter inside original')

        if (!range.collapsed) {
            document.execCommand('delete', false)

            range = null
            var selection = window.getSelection()
            if (selection && selection.rangeCount) {
                range = selection.getRangeAt(0)
            }

            if (range && !shouldSplitOriginalAtSelection(selection, range)) {
                // Oops
               return false
            }
        }

        if (range && range.startContainer && range.collapsed &&
            range.startContainer && range.startContainer.parentNode) {

            var startOffset = range.startOffset
            var node0, node1, curr

            var nodeType = range.startContainer.nodeType
            if (nodeType == Node.TEXT_NODE) {
                // Selection inside text node
                node0 = range.startContainer
                node1 = node0.splitText(startOffset)
                curr = node0.parentNode
            } else if (nodeType == Node.ELEMENT_NODE) {
                // Selection inside element node
                node0 = range.startContainer
                node1 = node0.cloneNode(false)

                var index = 0
                var child = node0.firstChild
                while (child != null) {
                    var next = child.nextSibling

                    if (index >= startOffset) {
                        node1.appendChild(child)
                    }

                    ++ index
                    child = next
                }

                curr = node0.parentNode
            }

            if (node0 == null || node1 == null || curr == null) {
                // Oops
                return false
            }

            while (true) {
                if (curr == null || curr == AQM_EDIT_AREA) {
                    break
                }

                var clone = curr.cloneNode(false)

                clone.appendChild(node1)

                var after = false
                var child = curr.firstChild
                while (child != null) {
                    var next = child.nextSibling

                    if (after) {
                        clone.appendChild(child)
                    } else if (child == node0) {
                        after = true
                    }

                    child = next
                }

                node0 = curr
                node1 = clone
                curr = curr.parentNode
            }

            var divNode = document.createElement('div')
            if (AQM_DEFAULT_BODY_STYLE) {
                setStyleCompat(divNode, AQM_DEFAULT_BODY_STYLE)
            }
            var breakNode = document.createElement('br')
            divNode.appendChild(breakNode)

            node0.parentNode.insertBefore(divNode, node0.nextSibling)
            node0.parentNode.insertBefore(node1, divNode.nextSibling)

            var r = document.createRange()
            r.setStartBefore(breakNode)
            r.collapse(true)
            // DEBUG
            // r.collapse(false)
            selection.removeAllRanges()
            selection.addRange(r)

            postSelectionCheck()

            return true
        }
    }

    return false
}

function splitOriginal() {
    // We can call this from debugger
    var selection = window.getSelection()
    if (selection && selection.rangeCount) {
        var range = selection.getRangeAt(0)
        if (range) {
            splitOriginalAtRange(range)
        }
    }
}

function onKeyDown(e) {
	if (e.which === 13) {
 		consoleLog('onKeyDown enter')

        var selection = window.getSelection()
        if (selection && selection.rangeCount) {
            var range = selection.getRangeAt(0)
            if (range && range.startContainer) {
                var special

                // Just before aqm-toggle
                if (isAtNodeWithId('aqm-toggle', range)) {
                    special = range.startContainer

                    consoleLog('Special case for enter before toggle ' + special.id)

                    var before = document.createElement('div')
                    var br = document.createElement('br')
                    before.appendChild(br)
                    special.parentNode.insertBefore(before, special)

                    var r = document.createRange()
                    r.selectNodeContents(before)
                    r.collapse(true)
                    selection.removeAllRanges()
                    selection.addRange(r)

                    postSelectionCheck()

                    range.detach()
                    r.detach()

                    e.stopPropagation()
                    e.preventDefault()
                    return false
                }

                // Special cases for greeting / signature and other special blocks, "at end of"
                special = isAtNodeTextEnd(AQM_SPECIAL_NAMES_FIX_AFTER_END, range)
                if (special) {
                    if (!range.collapsed) {
                        document.execCommand('delete', false)
                    }

                    consoleLog('Special case for enter after node ' + special.id)

                    var after = document.createElement('div')
                    var br = document.createElement('br')
                    after.appendChild(br)
                    special.parentNode.insertBefore(after, special.nextSibling)

                    var r = document.createRange()
                    r.selectNodeContents(after)
                    r.collapse(true)
                    selection.removeAllRanges()
                    selection.addRange(r)

                    postSelectionCheck()

                    range.detach()
                    r.detach()

                    e.stopPropagation()
                    e.preventDefault()
                    return false
                }

                // Special cases for "at start of"
                special = isAtNodeTextStart(AQM_SPECIAL_NAMES_FIX_BEFORE_START, range)
                if (special) {
                    if (!range.collapsed) {
                        document.execCommand('delete', false)
                    }

                    consoleLog('Special case for enter before node ' + special.id)

                    // Try to keep selection
                    var r = range.cloneRange()

                    var before = document.createElement('div')
                    var br = document.createElement('br')
                    before.appendChild(br)
                    special.parentNode.insertBefore(before, special)

                    selection.removeAllRanges()
                    selection.addRange(r)

                    postSelectionCheck()

                    range.detach()
                    r.detach()

                    e.stopPropagation()
                    e.preventDefault()
                    return false
                }

                // Special case for splitting the original message (inline replies)
                if (splitOriginalAtSelection(selection, range)) {

                    e.stopPropagation()
                    e.preventDefault()
                    return false
                }
            }

/*
            if (range && range.startContainer && range.startContainer == range.endContainer) {
                var node = range.startContainer

                consoleLog('start, end container = ' + range.startContainer + 'ancestor = ' +
                    range.commonAncestorContainer + ', parent = ' + node.parentNode)

                if (node.nodeType == Node.TEXT_NODE) {
                    var parent = node.parentNode
                    if (parent && parent.nodeType == Node.ELEMENT_NODE && parent.tagName == 'DIV') {
                        consoleLog('special case 1')
                        document.execCommand('insertParagraph')
                        postSelectionCheck()
                        return
                    }
                } else if (node.nodeType == Node.ELEMENT_NODE && node.tagName == 'DIV' &&
                        node.childElementCount == 1) {
                    var child = node.children[0]
                    if (child && child.nodeType == Node.ELEMENT_NODE && child.tagName == 'BR') {
                        consoleLog('special case 2')
                        document.execCommand('insertParagraph')
                        postSelectionCheck()
                        return
                    }
                }
            }
*/

            if (range) {
                range.detach()
            }
        }

/*
       consoleLog('using insertHtml')
       document.execCommand('insertHtml', false, '<div><br></div>')
       postSelectionCheck()
*/
    }

    return true
}

function onSelectionChanged() {
//    consoleLog('onSelectionChanged')
    postSelectionCheck()
}

// Cut / copy

function onCut(ev) {
    if (AQM_IS_RICH_FORMAT) {
        // HTML
        consoleLog('Cut - html text')
        return true
    } else {
        // Make sure to cut plain text
        consoleLog('Cut - plain text')

        var selection = document.getSelection()
        if (selection && selection.rangeCount > 0) {
            var text = selection.toString()
            ev.clipboardData.setData('text/plain', text);
        }

        // Remove
        doSimpleCommand('delete')

        // Do not propagate
        ev.stopPropagation()
        ev.preventDefault()
        return false
    }
}

function onCopy(ev) {
    if (AQM_IS_RICH_FORMAT) {
        // HTML
        consoleLog('Copy - html text')
        return true
   } else {
        // Make sure to copy plain text
        consoleLog('Copy - plain text')

        var selection = document.getSelection()
        if (selection && selection.rangeCount > 0) {
            var text = selection.toString()
            ev.clipboardData.setData('text/plain', text);
        }

        // Do not propagate
        ev.stopPropagation()
        ev.preventDefault()
        return false
   }
}

function getClipboardDataCompat(ev, mime) {
    var text = ev.clipboardData.getData(mime)
    if (!AQM_IS_NEW_CHROME_57 && (text == null || text == '')) {
        text = AQM_EDIT_CALLBACK.getClipboardData(mime)
    }
    return text
}

function tryPastePlainReallyHtml() {
    // The Java side will get plain text from the clipboard and decide if it looks like HTML
    var htmlText = AQM_EDIT_CALLBACK.getClipboardPlainReallyHtml()

    if (htmlText) {
        consoleLog('Paste - html from plain text')

        // This is able to parse body tags and will create one if not already there.
        // We need the "inside" of that
        var parser = new DOMParser()
        var fragment = parser.parseFromString(htmlText, 'text/html')

        if (fragment && fragment.body) {
            // Remove styles if any
            var listStyles = fragment.body.querySelectorAll('style')
            if (listStyles) {
                for (var i = listStyles.length - 1; i >= 0; -- i) {
                    var nodeStyle = listStyles[i]
                    nodeStyle.remove()
                }
            }

            // Use just the inside of body tag
            var htmlCooked = fragment.body.innerHTML
            if (htmlCooked) {
                doArgCommand('insertHtml', htmlCooked)
            }

            attachImageOnClick()

            disableForms()

            if (!AQM_IS_RICH_FORMAT) {
                AQM_IS_RICH_FORMAT = true
                AQM_EDIT_CALLBACK.onSwitchedToRichFormat()
            }

            // It worked
            return true
        }
    }
}

function onPaste(ev) {
    if (AQM_SIGNATURE_MODE) {
        // Special case - paste plain text which is really HTML
        if (tryPastePlainReallyHtml()) {

            // Yes it worked - done
            ev.stopPropagation()
            ev.preventDefault()
            return false
        }
    }

    if (AQM_IS_RICH_FORMAT) {
        // HTML
        consoleLog('Paste - html text')

        var htmlRaw = getClipboardDataCompat(ev, 'text/html')
        if (htmlRaw) {
            var htmlCooked = AQM_EDIT_CALLBACK.sanitizeHtmlForPaste(htmlRaw)
            if (htmlCooked) {
                doArgCommand('insertHtml', htmlCooked)

                attachImageOnClick()

                disableForms()

                ev.stopPropagation()
                ev.preventDefault()
                return false
            }
        }

        var text = getClipboardDataCompat(ev, 'text/plain')
        if (text) {
            doArgCommand('insertText', text)
        } else {
            checkPasteNothingBug()
        }

        ev.stopPropagation()
        ev.preventDefault()
        return false
   } else {
        // Make sure to paste plain text
        consoleLog('Paste - plain text')

        var text = getClipboardDataCompat(ev, 'text/plain')
        if (text) {
            doArgCommand('insertText', text)
        } else {
            checkPasteNothingBug()
        }

        ev.stopPropagation()
        ev.preventDefault()
        return false;
    }
}

// Mutation observer

var AQM_OBSERVER = null

function addDirAutoToDivs(nodeList) {
    var len = nodeList.length
    for (var i = 0; i < len; ++ i) {
        var node = nodeList[i]
        if (node.nodeType == Node.ELEMENT_NODE && node.tagName == 'DIV' &&
                !node.hasAttribute('dir')) {
            node.setAttribute('dir', 'auto')
        }
    }
}

function mutationObserverFunc(mutations) {
    if (mutations) {
        var len = mutations.length
        if (len > 0) {
            for (var i = 0; i < len; ++ i) {
                var mut = mutations[i]
                if (mut.addedNodes) {
                    addDirAutoToDivs(mut.addedNodes)
                }
            }

            postIsDirty()
        }
    }
}

function registerMutationObserver() {
    if (AQM_OBSERVER == null) {
        consoleLog('register mutation observer')

        AQM_OBSERVER = new MutationObserver(mutationObserverFunc)
        AQM_OBSERVER.observe(AQM_EDIT_AREA, { attributes: true, childList: true, characterData: true, subtree: true })
    }
}

function unregisterMutationObserver() {
    if (AQM_OBSERVER) {
        consoleLog('unregister mutation observer')

        AQM_OBSERVER.disconnect()
        AQM_OBSERVER = null
    }
}

// Is rich format?

var AQM_IS_RICH_FORMAT = false
var AQM_DOUBLE_LINE_FIX_REPLACEMENT = "{aqm_double&iquest;line&frac34;fix_replacement&frac12;text}"

function doSetIsRichFormat(isRichFormat) {
    if (AQM_IS_RICH_FORMAT != isRichFormat) {
        consoleLog('isRichFormat change: ' + isRichFormat)
        AQM_IS_RICH_FORMAT = isRichFormat

        if (AQM_ALLOW_RICH_TO_PLAIN && !AQM_IS_RICH_FORMAT) {
            // Selection
            var selection = document.getSelection()

            // Remove formatting when switching from html to plain text
            var textPlain = collectSaveTextPlain(AQM_EDIT_AREA, false)
            var textHtml = AQM_EDIT_CALLBACK.convertPlainTextToHtml(textPlain)

            // Remove existing content
            while (AQM_EDIT_AREA.lastChild) {
                AQM_EDIT_AREA.removeChild(AQM_EDIT_AREA.lastChild);
            }

            // Remove existing styles except our special edit styles
            var listOldStyle = document.querySelectorAll('style')
            if (listOldStyle) {
                for (var i = listOldStyle.length - 1; i >= 0; -- i) {
                    var nodeOldStyle = listOldStyle[i]
                    if (nodeOldStyle.id != 'aqm-web-editor-styles') {
                        nodeOldStyle.remove()
                    }
                }
            }

            // Set new content
            if (textHtml) {
               var fragment = createFragmentFromHtml(textHtml)
               if (fragment) {
                    // Move all children from the wrapper into our own div
                    while (fragment.firstChild) {
                        AQM_EDIT_AREA.appendChild(fragment.firstChild)
                    }
                }
            }

            // Selection
             if (selection) {
                var range = document.createRange()

                range.selectNodeContents(AQM_EDIT_AREA)
                range.collapse(true)

                selection.removeAllRanges()
                selection.addRange(range)
            }

            postSelectionCheck()
        }
    }
}

// Toggle (of original text)

function onToggleClick(ev) {
    consoleLog('toggle click')

    var target = ev.target
    if (target) {
        var rect = target.getBoundingClientRect()
        if (rect) {
            var offsetX = ev.clientX - rect.left
            var offsetY = ev.clientY - rect.top

            consoleLog('*** x = ' + offsetX + ', y = ' + offsetY)

            if (offsetY < 16 || offsetY > 48 || offsetX > 64) {
                consoleLog('*** did not hit the expand button')
                return
            }
        }
    }

    var original = AQM_ORIGINAL
    var toggle = ev.target
    if (original && toggle) {

        unregisterMutationObserver()

        toggle.parentNode.insertBefore(original, toggle)
        toggle.parentNode.removeChild(toggle)

        // Add new line above (or below) the now expanded original
        var wrapper = document.createElement('div')
        var breakNode = document.createElement('br')
        wrapper.setAttribute('dir', 'auto')
        wrapper.appendChild(breakNode)

        var parent = original.parentNode
        var newline = document.createTextNode('\n')

        if (AQM_REPLY_INLINE) {
            var next = original.nextSibling
            if (next) {
                parent.insertBefore(wrapper, next)
                parent.insertBefore(newline, next)
            } else {
                parent.appendChild(wrapper)
                parent.appendChild(newline)
            }
        } else {
            parent.insertBefore(wrapper, original)
            parent.insertBefore(newline, original)
        }

        AQM_ORIGINAL = null
        toggle = null

        // Select that new line
        var selection = document.getSelection()
        if (selection) {
            var range = document.createRange()

            range.setStartBefore(breakNode)
            range.collapse(true)

            selection.removeAllRanges()
            selection.addRange(range)
        }

        attachImageOnClick()

        registerMutationObserver()
    }
}

// Init

function initialize() {
    consoleLog('editor init')

    AQM_EDIT_AREA = document.getElementById('aqm-editor')
    consoleLog('edit area: ' + AQM_EDIT_AREA)

    // String.endsWith polyfill
    if (typeof(String.prototype.endsWith) !== 'function') {
        consoleLog('polyfill for String.endsWith')
        String.prototype.endsWith = function(suffix) {
            return this.indexOf(suffix, this.length - suffix.length) !== -1
        }
    }

    // String.startsWith polyfill
    if (typeof(String.prototype.startsWith) !== 'function') {
        consoleLog('polyfill for String.startsWith')
        String.prototype.startsWith = function(suffix) {
            var thisStr = String(this)
            var thisLen = thisStr.length
            var suffixStr = String(suffix)
            var suffixLen = suffixStr.length

            if (thisLen < suffixLen) {
                return false
            }

            for (var i = 0; i < suffixLen; ++ i) {
                if (thisStr.charCodeAt(i) != suffixStr.charCodeAt(i)) {
                    return false
                }
            }

            return true
        }
    }

    // We want "div" for line breaks - "p" add extra newlines
    document.execCommand('defaultParagraphSeparator', false, 'div')

    // Default style
    if (AQM_DEFAULT_BODY_STYLE) {
        consoleLog('default style: ' + AQM_DEFAULT_BODY_STYLE)
        setStyleCompat(AQM_EDIT_AREA, AQM_DEFAULT_BODY_STYLE)
    }

    // Key down for better "newline" handling
    document.addEventListener('keydown', onKeyDown)

    // Selection change monitoring
    document.addEventListener('selectionchange', onSelectionChanged)

    // Cut / copy
    AQM_EDIT_AREA.oncut = onCut
    AQM_EDIT_AREA.oncopy = onCopy
    AQM_EDIT_AREA.onpaste = onPaste

    // Mutations (changes)
    registerMutationObserver()

    // Init is done
    consoleLog('AQM_EDIT_CALLBACK = ' + AQM_EDIT_CALLBACK)
    AQM_EDIT_CALLBACK.initIsDone()
}

initialize()

// Functions for Java side

function doSimpleCommand(name) {
    consoleLog('execCommand ' + name)
    document.execCommand(name)
    postSelectionCheck()
}

function doArgCommand(name, arg) {
    consoleLog('execCommand ' + name + ', ' + arg)
    document.execCommand(name, false, arg)
    postSelectionCheck()
}

function doBold() {
    doSimpleCommand('bold')
}

function doItalic() {
    doSimpleCommand('italic')
}

function doUnderline() {
    doSimpleCommand('underline')
}

function doStrikethrough() {
    doSimpleCommand('strikethrough')
}

function doForeColor(color) {
    doArgCommand('forecolor', color)
}

function doBackColor(color) {
    doArgCommand('backcolor', color)
}

function doFontSize(size) {
    doArgCommand('fontSize', size)
}

function doFontName(name) {
    doArgCommand('fontName', name)
}

function doClearFormat() {
    doSimpleCommand('removeFormat')
}

function doSetLink(link) {
    var url = safeDecodeSnippet(link)
    consoleLog('execCommand doSetLink ' + url)

    if (url) {
        selectSelectedAnchor()
        doArgCommand('createLink', url)
    }
}

function doClearLink() {
    consoleLog('execCommand doClearLink')

    selectSelectedAnchor()
    doSimpleCommand('unlink')
}

function doParagraphAlign(align) {
    consoleLog('execCommand doParagraphAlign ' + align)

    var command
    if (align == AQM_STATE_ALIGN_LEFT) {
        command = 'justifyLeft'
    } else if (align == AQM_STATE_ALIGN_RIGHT) {
        command = 'justifyRight'
    } else if (align == AQM_STATE_ALIGN_CENTER) {
        command = 'justifyCenter'
    } else if (align == AQM_STATE_ALIGN_JUSTIFY) {
        command = 'justifyFull'
    }

    if (command) {
        doSimpleCommand(command)
    }
}

function doListBulleted() {
    consoleLog('doListBulleted')

    if (!document.queryCommandState('insertUnorderedList')) {
        doSimpleCommand('insertUnorderedList')
    }
}

function doListNumbered() {
    consoleLog('doListNumbered')

    if (!document.queryCommandState('insertOrderedList')) {
        doSimpleCommand('insertOrderedList')
    }
}

function doIndentDecrease() {
    consoleLog('doIndentDecrease')
    doSimpleCommand('outdent')
}

function doIndentIncrease() {
    consoleLog('doIndentIncrease')
    doSimpleCommand('indent')
}

/*

function doBlockQuoteToggle() {
    consoleLog('execCommand blockquote toggle')

    var ancestorEl
    var selection = document.getSelection()
    if (selection && selection.rangeCount > 0) {
        var range = selection.getRangeAt(0)
        if (range) {
            ancestorEl = getRangeAncestorContainer(range)
            var blockQuote = findEnclosingBlock(ancestorEl, 'blockquote')
            if (blockQuote == ancestorEl) {
                // We are on a block quote right now
                consoleLog('on a blockquote now')
                doSimpleCommand('outdent')
            } else if (blockQuote) {
                // Remove blockquote by first moving selection to it
                consoleLog('found a blockquote above')
//                    var rangeBQ = document.createRange()
//
//                    rangeBQ.selectNode(blockQuote)
//
//                    selection.removeAllRanges()
//                    selection.addRange(rangeBQ)

                doSimpleCommand('outdent')

                // At this point we don't have anything to set selection to
                // selection.removeAllRanges()
            } else {
                // Make blockquote
                consoleLog('no blockquote, creating')
                doArgCommand('formatblock', 'blockquote')
            }
        }
    }
}

function doUnorderedListToggle() {
    consoleLog('execCommand unordered list toggle')
    doSimpleCommand('insertUnorderedList')
}

*/

function doUndo() {
    consoleLog('execCommand undo')
    doSimpleCommand('undo')
}

function doRedo() {
    consoleLog('execCommand redo')
    doSimpleCommand('redo')
}

// Images

function randomImageId() {
    return Math.random().toString(36).substring(2) + (new Date()).getTime().toString(36)
}

function attachImageOnClick() {
    var list = document.querySelectorAll('img[data-aqm-editable]')
    for (var i = 0; i < list.length; ++ i) {
        var node = list[i]
        if (node.onclick == null) {
            node.onclick = onImageClick
        }
   }
}

function doInsertImage(contentId, encoded) {
    var cid = decodeURIComponent(contentId)
    var uri = decodeURIComponent(encoded)
    consoleLog('doInsertImage: ' + uri)

    ensureSelectBeforeInsert()

    var style = 'height: auto; border:none;'
    var html = '<img style=\'' + style + '\' data-aqm-cid=\'' + cid +
        '\' data-aqm-editable=\'' + true + '\' src=\'' + uri + '\'>'

    doArgCommand('insertHtml', html)

    attachImageOnClick()
}

function findImageByEditId(imageId, relaxed) {
    if (imageId) {
        var decoded = decodeURIComponent(imageId)
        var list = document.querySelectorAll('img[data-aqm-image-id]')
        for (var i = 0; i < list.length; ++ i) {
            var node = list[i]
            if (node.getAttribute('data-aqm-image-id') == decoded) {
                if (relaxed) {
                    return node
                }
                if (node.complete && node.naturalWidth > 0 && node.naturalHeight > 0) {
                    return node
                }
                return null
            }
        }
    }
    return null
}

function getImageLinkParentNode(target) {
    var parent = target.parentNode
    if (parent.tagName == 'A' && parent.childElementCount == 1) {
        return parent
    }
    return null
}

function selectAfterImage(target) {
    var selection = document.getSelection()
    var range = document.createRange()

    var node = target
    var parent = getImageLinkParentNode(target)
    if (parent) {
        node = parent
    }

    range.setStartAfter(node)
    range.setEndAfter(node)
    selection.removeAllRanges()
    selection.addRange(range)
}

function onImageClick(event) {
    if (!event) {
        return
    }

    var target = event.target
    if (!target || target.nodeType != Node.ELEMENT_NODE) {
        return
    }

    var randomId = randomImageId()
    consoleLog('onImageClick: target = ' + target + ', id = ' + randomId)

    if (target.complete && target.naturalWidth > 0 && target.naturalHeight > 0) {
        unregisterMutationObserver()

        target.setAttribute('data-aqm-image-id', randomId)

        selectAfterImage(target)

        var curWidth = target.style['width'] || target.getAttribute('width')
        var curAlign = target.style['vertical-align']

        var x = 0, y = 0
        var rect = target.getBoundingClientRect()
        if (rect) {
            x = rect.left + (rect.right - rect.left) / 4
            y = rect.top
        }

        var link = null
        var parent = getImageLinkParentNode(target)
        if (parent) {
            link = parent.getAttribute('href')
        }

        AQM_EDIT_CALLBACK.showImageMenu(randomId, curWidth, target.naturalWidth, curAlign, x, y, link)

        registerMutationObserver()
    }
}

function doChangeImageWidth(imageId, widthFraction) {
    var target = findImageByEditId(imageId)
    consoleLog('doChangeImageSize: imageId = ' + imageId + ', node = ' + target)

    if (target) {
        var widthPx = (widthFraction * target.naturalWidth + 0.5) | 0
        if (widthPx >= target.naturalWidth) {
            target.style['width'] = null
        } else {
            target.style['width'] = widthPx + 'px'
        }
        target.style['height'] = 'auto'

        target.removeAttribute('width')
        target.removeAttribute('height')

        selectAfterImage(target)
    }
}

function doSetImageAlign(imageId, align) {
    var target = findImageByEditId(imageId)

    consoleLog('doSetImageAlign: imageId = ' + imageId + ', node = ' + target)
    if (target) {
        target.style['vertical-align'] = align

        selectAfterImage(target)
    }
}

function doSetImageLink(imageId, link) {
    var target = findImageByEditId(imageId)

    consoleLog('doSetImageAlign: imageId = ' + imageId + ', node = ' + target)
    if (target) {
        var href = safeDecodeSnippet(link)
        var parent = getImageLinkParentNode(target)

        var selection = document.getSelection()
        var range = document.createRange()

        if (href) {
            // Set link
            if (parent) {
                parent.setAttribute('href', href)
            } else {
                var after = document.createRange()
                after.setStartAfter(target)
                after.setEndAfter(target)

                range.selectNode(target)
                selection.removeAllRanges()
                selection.addRange(range)

                var html = '<a href="' + escapeHtmlString(href) + '">' + target.outerHTML + '</a>'
                doArgCommand('insertHtml', html)

                selection.removeAllRanges()
                selection.addRange(after)

                // Attach image editing
                attachImageOnClick()
            }
        } else {
            // Clear link
            if (parent) {
                var after = document.createRange()
                after.setStartAfter(parent)
                after.setEndAfter(parent)

                range.selectNode(parent)
                selection.removeAllRanges()
                selection.addRange(range)

                var html = target.outerHTML
                doArgCommand('insertHtml', html)

                selection.removeAllRanges()
                selection.addRange(after)
            }
        }
    }
}

function doDeleteImage(imageId) {
    var target = findImageByEditId(imageId, true)

    consoleLog('doDeleteImage: imageId = ' + imageId + ', node = ' + target)
    if (target) {
        var parent = getImageLinkParentNode(target)

        var selection = document.getSelection()
        var range = document.createRange()
        var after = document.createRange()

        if (parent) {
            range.selectNode(parent)
            after.setStartAfter(parent)
            after.setEndAfter(parent)
        } else {
            range.selectNode(target)
            after.setStartAfter(target)
            after.setEndAfter(target)
        }
        selection.removeAllRanges()
        selection.addRange(range)

        var html = ''
        doArgCommand('insertHtml', html)

        selection.removeAllRanges()
        selection.addRange(after)
    }
}

// Editor content

function doGetEditContent() {
    var html = AQM_EDIT_AREA.innerHTML
    AQM_EDIT_CALLBACK.thisIsContent(html)
}

function doSetEditContent(seed, encodedStyle, needToggle) {
    consoleLog('doSetEditContent seed = ' + seed)

    var html = AQM_EDIT_CALLBACK.getLoadContent(seed)
    var decodedStyle = safeDecodeSnippet(encodedStyle)

    if (html != null) {
        // Unregister mutation observer
        unregisterMutationObserver()

//        var fragment = document.createRange().createContextualFragment(html)

        // This is able to parse body tags, we don't need that, but... it already worked
        var parser = new DOMParser()
        var fragment = parser.parseFromString(html, 'text/html')

        consoleLog('doSetEditContent 2: ' + fragment)

        if (fragment) {
            // Remove existing content
            while (AQM_EDIT_AREA.lastChild) {
                AQM_EDIT_AREA.removeChild(AQM_EDIT_AREA.lastChild);
            }

            // Remove existing styles except our special edit styles
            var listOldStyle = document.querySelectorAll('style')
            if (listOldStyle) {
                for (var i = listOldStyle.length - 1; i >= 0; -- i) {
                    var nodeOldStyle = listOldStyle[i]
                    if (nodeOldStyle.id != 'aqm-web-editor-styles') {
                        nodeOldStyle.remove()
                    }
                }
            }

            // Content specific style
            if (decodedStyle) {
                consoleLog('default style: ' + AQM_DEFAULT_BODY_STYLE)
                AQM_DEFAULT_BODY_STYLE = decodedStyle
                setStyleCompat(AQM_EDIT_AREA, AQM_DEFAULT_BODY_STYLE)
            }

            // Find and append style tags into <head>
            var listNewStyle = fragment.querySelectorAll('style')
            if (listNewStyle) {
                var listNewStyleSize = listNewStyle.length
                consoleLog('doSetEditContent style list: ' + listNewStyleSize)

                var head = document.getElementById('aqm-web-editor-head')

                for (var i = 0; i < listNewStyleSize; ++ i) {
                    var nodeNewStyle = listNewStyle[i]
                    head.appendChild(nodeNewStyle)
                }
            }

            // Editable wrapper
            var nodeEditable = fragment.getElementById('aqm-editable')
            if (nodeEditable) {
                consoleLog('doSetEditContent editable wrapper: ' + nodeEditable)

                // Fix paragraphs
                if (AQM_SIGNATURE_MODE) {
                    fixParagraphs(fragment)
                }

                // Replace original with toggle, setting the original aside
                var toggle = null
                AQM_ORIGINAL = null
                if (needToggle) {
                    var original = fragment.getElementById('aqm-original')
                    if (original) {
                        toggle = fragment.createElement('div')
                        toggle.id = 'aqm-toggle'
                        toggle.className = 'aqm-toggle'
                        toggle.setAttribute('contenteditable', false)
                        // toggle.innerText = '\u2026'

                        original.parentNode.insertBefore(toggle, original)
                        original.parentNode.removeChild(original)

                        AQM_ORIGINAL = original
                        toggle.onclick = onToggleClick
                    }
                }

                // Move all children from the wrapper into our own div
                while (nodeEditable.firstChild) {
                    AQM_EDIT_AREA.appendChild(nodeEditable.firstChild)
                }

                // Disable forms
                disableForms()

                // Ensure blank space before and after toggle so it can be selected and backspaced
                if (toggle) {
                    if (!AQM_REPLY_INLINE) {
                        ensureLineBreakBefore(toggle)
                    }
                    ensureLineBreakAfter(toggle)
                }
            } else {
                consoleLog('doSetEditContent editable wrapper: null')
            }

            // Attach image editing
            attachImageOnClick()
        }

        // Register mutation observer again
        registerMutationObserver()

        consoleLog('doSetEditContent done')
    } else {
        consoleLog('doSetEditContent did not get data')
    }
}

// Greeting and signature

function collectInnerText(node) {
    var text = ''
    if (node.children) {
        for (var i = 0; i < node.children.length; ++ i) {
            var child = node.children[i]
            text += child.innerText.trim()
        }
    }
    return text.replace(/\s/mg, '')
}

function isSpecialIdDiv(node) {
    var id = node.id
    if (id) {
        for (var i = 0; i < AQM_SPECIAL_NAMES_ALL.length; ++ i) {
            if (id == AQM_SPECIAL_NAMES_ALL[i]) {
                return true
            }
        }
    }
    return false
}

function getLineBreakAfter(node) {
    var next = node.nextElementSibling
    if (next) {
       if (next.nodeType == Node.ELEMENT_NODE) {
            if (next.tagName == 'BR') {
                return next
            } else if (next.tagName == 'DIV' && !isSpecialIdDiv(next)) {
                var first = next.firstChild
                if (first) {
                    if (first.nodeType == Node.TEXT_NODE && first.nodeValue.trim() == '') {
                        first = last.nextSibling
                    }

                    if (first && first.nodeType == Node.ELEMENT_NODE && first.tagName == 'BR') {
                        return first
                    }
                }
            }
        }
    }

    return null
}

function ensureLineBreakAfter(node, text) {
    var after = getLineBreakAfter(node)
    if (!after) {
        consoleLog('Adding line break after ' + text)

        var wrapper = document.createElement('div')
        after = document.createElement('br')
        wrapper.setAttribute('dir', 'auto')
        wrapper.appendChild(after)

        node.parentNode.insertBefore(wrapper, node.nextSibling)
    }

    return after
}

function getLineBreakBefore(node) {
    var prev = node.previousElementSibling
    if (prev) {
        if (prev.nodeType == Node.ELEMENT_NODE) {
            if (prev.tagName == 'BR') {
                return prev
            } else if (prev.tagName == 'DIV' && !isSpecialIdDiv(prev)) {
                var last = prev.lastChild
                if (last) {
                    if (last.nodeType == Node.TEXT_NODE && last.nodeValue.trim() == '') {
                        last = last.previousSibling
                    }

                    if (last && last.nodeType == Node.ELEMENT_NODE && last.tagName == 'BR') {
                        return last
                    }
                }
            }
        }
    }
    return null
}

function ensureLineBreakBefore(node, text) {
    var before = getLineBreakBefore(node)
    if (!before) {
        consoleLog('Adding line break before ' + text)

        var wrapper = document.createElement('div')
        before = document.createElement('br')
        wrapper.setAttribute('dir', 'auto')
        wrapper.appendChild(before)

        node.parentNode.insertBefore(wrapper, node)
    }

    return before
}

function createFragmentFromHtml(html) {
    if (AQM_IS_NEW_CHROME_57) {
        return document.createRange().createContextualFragment(html)
    } else {
        consoleLog('Fallback for createContextualFragment')
        var div = document.createElement('div')
        div.innerHTML = html
        return div
    }
}

function fixParagraphs(fragment) {
    var plist = fragment.querySelectorAll('p')
    if (plist) {
        var list = copyNodeList(plist)
        var len = list.length
        for (var i = 0; i < len; ++ i) {
            var p = list[i]
            if (p.style && p.style.marginBottom == '1em') {
                p.style.marginBottom = '0px'

                if (!isLastNodeInBlock(p) && p.parentNode) {
                    var spacer = document.createElement('div')
                    spacer.setAttribute('dir', 'auto')
                    spacer.appendChild(document.createElement('br'))

                    p.parentNode.insertBefore(spacer, p.nextSibling)

                    if (p.firstElementChild == null && areAllTextChildrenWhitespace(p)) {
                        p.parentNode.removeChild(p)
                    }
                }
            }
        }
    }
}

function doInsertGreeting(isRichFormat, encodedHtml, encodedPlain, addWhiteSpace) {
    var html = safeDecodeSnippet(encodedHtml)
    var plain = safeDecodeSnippet(encodedPlain)
    consoleLog('doInsertGreeting html: ' + html + ', plain: ' + plain +
        ', addWhiteSpace ' + addWhiteSpace)

    if (html || plain) {
        // Unregister mutation observer
        unregisterMutationObserver()

        // Create wrapper
        var wrapper = document.getElementById('aqm-greeting')
        if (!wrapper) {
            wrapper = document.createElement('div')
            wrapper.id = 'aqm-greeting'
            wrapper.setAttribute('dir', 'auto')
            if (AQM_DEFAULT_WRAPPER_STYLE) {
                setStyleCompat(wrapper, AQM_DEFAULT_WRAPPER_STYLE)
            }
            AQM_EDIT_AREA.insertBefore(wrapper, AQM_EDIT_AREA.firstChild)
        }

        if (AQM_REPLY_INLINE) {
            if (document.getElementById('aqm-toggle') != null || document.getElementById('aqm-original') != null) {
                addWhiteSpace = false
            }
        }

        if (addWhiteSpace) {
            ensureLineBreakAfter(wrapper, 'greeting wrapper')
        }

        if (isRichFormat) {
            if (html) {
                var fragment = createFragmentFromHtml(html)
                if (fragment) {
                    var fragmentText = collectInnerText(fragment)
                    var wrapperText = collectInnerText(wrapper)

                    consoleLog('Fragment text: ' + fragmentText)
                    consoleLog('Wrapper text: ' + wrapperText)

                    if (wrapper.childNodes.length == 0 || wrapperText == '' || !wrapperText.startsWith(fragmentText)) {
                        fixParagraphs(fragment)

                        var prev = null
                        var list = copyNodeList(fragment.children)
                        for (var i = 0; i < list.length; ++ i) {
                            var item = list[i]
                            if (prev == null) {
                                wrapper.insertBefore(item, wrapper.firstChild)
                            } else {
                                wrapper.insertBefore(item, prev.nextSibling)
                            }
                            prev = item
                        }
                    }

                    // Attach image editing
                    attachImageOnClick()
                }
            }
        } else {
            if (plain) {
                var wrapperTextOld = wrapper.innerText
                var wrapperTextNew = null

                consoleLog('New text: ' + plain)
                consoleLog('Old text: ' + wrapperTextOld)

                if (wrapperTextOld == '') {
                    wrapperTextNew = plain
                } else if (!wrapperTextOld.startsWith(plain)) {
                    wrapperTextNew = plain
                    if (!wrapperTextNew.endsWith('\n')) {
                        wrapperTextNew += '\n'
                    }
                    wrapperTextNew += wrapperTextOld
                }
                if (wrapperTextNew) {
                    wrapper.innerText = wrapperTextNew
                }
            }
        }

        // Register mutation observer again
        registerMutationObserver()
    }
}

function doRemoveGreeting(encoded) {
    var html = safeDecodeSnippet(encoded)
    consoleLog('doRemoveGreeting ' + html)

    var wrapper = document.getElementById('aqm-greeting')
    if (wrapper) {
        // Unregister mutation observer
        unregisterMutationObserver()

        var fragment = createFragmentFromHtml(html)
        if (fragment) {
            var fragmentText = collectInnerText(fragment)
            var wrapperText = collectInnerText(wrapper)

            consoleLog('Fragment text: ' + fragmentText)
            consoleLog('Wrapper text: ' + wrapperText)

            if (wrapperText == '' || wrapperText == fragmentText) {
                wrapper.innerHTML = ''
            }
        }

        // Register mutation observer again
        registerMutationObserver()
    }
}

var AQM_MOVE_BLANK_LINES = false

function doAppendSignature(isRichFormat, encodedHtml, encodedPlain, addWhiteSpace) {
    var html = safeDecodeSnippet(encodedHtml)
    var plain = safeDecodeSnippet(encodedPlain)
    consoleLog('doAppendSignature html: ' + html + ', plain: ' + plain +
        ', addWhiteSpace: ' + addWhiteSpace)

    if (html || plain) {
        // Unregister mutation observer
        unregisterMutationObserver()

        // Create wrapper
        var wrapper = document.getElementById('aqm-signature')
        if (!wrapper) {
            wrapper = document.createElement('div')
            wrapper.id = 'aqm-signature'
            wrapper.setAttribute('dir', 'auto')

            if (AQM_DEFAULT_WRAPPER_STYLE) {
                setStyleCompat(wrapper, AQM_DEFAULT_WRAPPER_STYLE)
            }

            if (AQM_REPLY_INLINE) {
                // Reply along the bottom
                AQM_EDIT_AREA.appendChild(wrapper)
            } else if (AQM_SIGNATURE_MIDDLE) {
                // Keep signature in the middle
                var toggle = document.getElementById('aqm-toggle')
                if (toggle) {
                    toggle.parentNode.insertBefore(wrapper, toggle)
                } else {
                    var original = document.getElementById('aqm-original')
                    if (original) {
                        original.parentNode.insertBefore(wrapper, original)
                    }
                }
            }

            if (wrapper.parentNode == null) {
                AQM_EDIT_AREA.appendChild(wrapper)
            }
        }

        if (addWhiteSpace) {
            if (!AQM_REPLY_INLINE && !AQM_SIGNATURE_MIDDLE) {
                ensureLineBreakAfter(wrapper, 'signature wrapper')
            }
            ensureLineBreakBefore(wrapper, 'signature wrapper')
        }

        if (isRichFormat) {
            if (html) {
                var fragment = createFragmentFromHtml(html)
                if (fragment) {
                    var fragmentText = collectInnerText(fragment)
                    var wrapperText = collectInnerText(wrapper)

                    consoleLog('Fragment text: ' + fragmentText)
                    consoleLog('Wrapper text: ' + wrapperText)

                    if (wrapper.childNodes.length == 0 || wrapperText == '' || !wrapperText.endsWith(fragmentText)) {
                        fixParagraphs(fragment)

                        var list = copyNodeList(fragment.children)

                        // AQM-1439 - move blank lines at the start of the signature to be above it
                        if (AQM_MOVE_BLANK_LINES) {
                            for (var i = 0; i < list.length; ++ i) {
                                var item = list[i]
                                if (isDivNode(item)) {
                                    if (isDivNodeLineBreak(item)) {
                                        list[i] = null
                                        wrapper.parentNode.insertBefore(item, wrapper)
                                        AQM_MOVE_BLANK_LINES = false
                                    } else if (i == 0) {
                                        var curr = item.firstChild
                                        while (curr) {
                                            var next = curr.nextSibling

                                            if (isDivNode(curr) && isDivNodeLineBreak(curr)) {
                                                wrapper.parentNode.insertBefore(curr, wrapper)
                                                AQM_MOVE_BLANK_LINES = false
                                            } else if (!isTextNodeCanSkip(curr)) {
                                                break
                                            }

                                            curr = next
                                        }
                                    }
                                } else if (!isTextNodeCanSkip(item)) {
                                    break
                                }
                            }
                        }

                        for (var i = 0; i < list.length; ++ i) {
                            var item = list[i]
                            if (item) {
                                wrapper.appendChild(item)
                            }
                        }
                    }

                    // Attach image editing
                    attachImageOnClick()
                }
            }
        } else {
            if (plain) {
                var wrapperTextOld = wrapper.innerText
                var wrapperTextNew = null

                consoleLog('New text: ' + plain)
                consoleLog('Old text: ' + wrapperTextOld)

                // AQM-1439 - move blank lines at the start of the signature to be above it
                if (AQM_MOVE_BLANK_LINES) {
                    var len = plain.length
                    var pos = 0
                    while (pos < len && plain.charAt(pos) == '\n') {
                        ++ pos
                    }

                    if (pos > 0) {
                        plain = plain.substring(pos)

                        while (pos > 0) {
                            var curr = document.createElement('div')
                            curr.setAttribute('dir', 'auto')
                            curr.appendChild(document.createElement('br'))

                            wrapper.parentNode.insertBefore(curr, wrapper)

                            -- pos
                        }

                        AQM_MOVE_BLANK_LINES = false
                    }
                }

                if (wrapperTextOld == '') {
                    wrapperTextNew = plain
                } else if (!wrapperTextOld.endsWith(plain)) {
                    wrapperTextNew = wrapperTextOld
                    if (!wrapperTextNew.endsWith('\n')) {
                        wrapperTextNew += '\n'
                    }
                    wrapperTextNew += plain
                }
                if (wrapperTextNew) {
                    wrapper.innerText = wrapperTextNew
                }
            }
        }

        // Register mutation observer again
        registerMutationObserver()
   }
}

function doRemoveSignature(encoded) {
    var html = safeDecodeSnippet(encoded)
    consoleLog('doRemoveSignature ' + html)

    var wrapper = document.getElementById('aqm-signature')
    if (wrapper) {
        // Unregister mutation observer
        unregisterMutationObserver()

        var fragment = createFragmentFromHtml(html)
        if (fragment) {
            var fragmentText = collectInnerText(fragment)
            var wrapperText = collectInnerText(wrapper)

            consoleLog('Fragment text: ' + fragmentText)
            consoleLog('Wrapper text: ' + wrapperText)

            if (wrapperText == '' || wrapperText == fragmentText) {
                wrapper.innerHTML = ''
            }
        }

        // Register mutation observer again
        registerMutationObserver()
    }
}

// Insert quick response

function doInsertQuickResponse(encoded) {
    var text = safeDecodeSnippet(encoded)
    consoleLog('doInsertQuickResponse ' + text)

    if (text) {
        ensureSelectBeforeInsert()

        doArgCommand('insertText', text)
    }
}

function doSetInitialFocus() {
    consoleLog('doSetInitialFocus')

    unregisterMutationObserver()

    var breakNode, special

    if (AQM_REPLY_INLINE) {
        if ((special = document.getElementById('aqm-signature')) != null) {
            breakNode = ensureLineBreakBefore(special)
        } else if ((special = document.getElementById('aqm-toggle')) != null) {
            breakNode = ensureLineBreakAfter(special)
        } else if ((special = document.getElementById('aqm-greeting')) != null) {
            breakNode = ensureLineBreakAfter(special)
        }
    } else {
        if ((special = document.getElementById('aqm-greeting')) != null) {
            breakNode = ensureLineBreakAfter(special)
        } else if (AQM_SIGNATURE_MIDDLE && (special = document.getElementById('aqm-signature')) != null) {
            breakNode = ensureLineBreakBefore(special)
        } else if ((special = document.getElementById('aqm-toggle')) != null) {
            breakNode = ensureLineBreakBefore(special)
        } else if ((special = document.getElementById('aqm-original')) != null) {
            breakNode = ensureLineBreakBefore(special)
        } else if (!AQM_SIGNATURE_MIDDLE && (special = document.getElementById('aqm-signature')) != null) {
            breakNode = ensureLineBreakBefore(special)
        }
    }

    if (breakNode) {
        var selection = document.getSelection()
        var range = document.createRange()

        range.setStartBefore(breakNode)
        range.collapse(true)

        selection.removeAllRanges()
        selection.addRange(range)

        consoleLog('doSetInitialFocus - is done')

        AQM_EDIT_AREA.focus()

        AQM_EDIT_CALLBACK.onInitialFocusIsDone()
    }

    registerMutationObserver()
}

function doSetNormalFocus() {
    consoleLog('doSetNormalFocus')

    var selection = document.getSelection()
    if (selection && selection.rangeCount <= 0 && AQM_EDIT_AREA) {
//        var range = document.createRange()
//        range.selectNodeContents(AQM_EDIT_AREA)
//        range.collapse(true)
//
//        selection.removeAllRanges()
//        selection.addRange(range)

        AQM_EDIT_AREA.focus()
        AQM_EDIT_CALLBACK.onInitialFocusIsDone()
    }
}

// Saving

function needLineBreakAfterElementSubNode(sub) {
    if (sub.nodeType == Node.ELEMENT_NODE && sub.id != 'aqm-toggle' &&
            (AQM_BLOCK_TAGS_MAP[sub.tagName] || sub.tagName == 'BR')) {
        var last = sub.lastElementChild
        if (last && (AQM_BLOCK_TAGS_MAP[last.tagName] || last.tagName == 'BR')) {
        } else {
            return true
        }
    }
    return false
}

function isBlockElement(node) {
    return AQM_BLOCK_TAGS_MAP[node.tagName]
}

function isBlockOrBreakElement(node) {
    return AQM_BLOCK_TAGS_MAP[node.tagName] || node.tagName == 'BR'
}

function rc_revertBr(node) {
	var children = node.childNodes
    if (children.length == 0) {
    	return
    }

    var i;


	for (i = 0; i < children.length; i++) {
        var child = children[i]
        if (isDivNode(child)) {
        	rc_revertBr(child)
            continue
		}
		if (isSpanNode(child) && child.getAttribute("aqmreplace") == "from_br") {
           var br = document.createElement("BR")
		   node.replaceChild(br, child)
    	}
	}
}

function rc_replaceBr_start(node) {
    rc_replaceBr(node, true)
}

function rc_replaceBr(node, isFirst) {
	var children = node.childNodes
    if (children.length == 0) {
    	return
    }

    var i;


	for (i = 0; i < children.length; i++) {
        var child = children[i]
        if (isDivNode(child)) {
            rc_replaceBr(child, false)
            continue
		}
		if (!isFirst && children.length == 1 && isBrNode(child)) {
        	var span = document.createElement("SPAN")
            span.setAttribute("aqmreplace", "from_br")
            span.innerText = AQM_DOUBLE_LINE_FIX_REPLACEMENT
            node.replaceChild(span, child)
    	}
	}
}

function collectSaveTextPlainImpl(node, addQuotingPrefix, canUseInnerText) {
    var text = ''
    var nodeType = node.nodeType
    if (nodeType == Node.ELEMENT_NODE && node.id == 'aqm-toggle') {
        if (AQM_ORIGINAL) {
            // The innerText property is computed by layout.
            //
            // We keep original text out of the DOM tree until / if the user expands it, and until that happens,
            // innerText contains extra whitespace (spaces / line breaks), HTML whitespace treatment rules have
            // not been applied yet. Essentially we get textContent instead of innerText.
            //
            // And so, for plain text messages, we save the original plain text content exactly as is and use that.
            //
            // For HTML messages, preserving plain text "to the letter" is less important

            if (!AQM_REPLY_INLINE) {
                text += '\n'
            }

            var originalPlain = AQM_EDIT_CALLBACK.getOriginalPlain()
            if (originalPlain != null) {
                text += originalPlain
            } else {
                text += collectSaveTextPlainImpl(AQM_ORIGINAL, addQuotingPrefix, false)
            }

            if (!text.endsWith('\n')) {
                text += '\n'
            }

            if (AQM_REPLY_INLINE) {
                text += '\n'
            }
        }
    } else if (nodeType == Node.ELEMENT_NODE || nodeType == Node.DOCUMENT_NODE ||
            nodeType == Node.DOCUMENT_FRAGMENT_NODE) {
        if (canUseInnerText && typeof(node.innerText) === "string" &&
                querySelectorCompat(node, 'blockquote[type=cite i]', 'blockquote[type=cite]') == null &&
                (AQM_ORIGINAL == null || node.querySelector('.aqm-toggle') == null)) {
            // Hooray we can just use innerText

            unregisterMutationObserver()
            rc_replaceBr_start(node)

            var textPlain = node.innerText;
            textPlain = textPlain.replace(new RegExp(AQM_DOUBLE_LINE_FIX_REPLACEMENT, 'g'), "")
            text = textPlain

            rc_revertBr(node)
            registerMutationObserver()
        } else if (node.firstElementChild && areAllTextChildrenWhitespace(node)) {
            // All text node children contain only whitespace, we can only look at element nodes

            for (var sub = node.firstElementChild; sub; sub = sub.nextElementSibling) {
                var s = collectSaveTextPlainImpl(sub, addQuotingPrefix, canUseInnerText)
                if (needLineBreakAfterElementSubNode(sub)) {
                    s += '\n'
                }
                text += s
            }
        } else if (node.firstChild) {
            // There are some text node children that contain meaningful text (not just whitespace)
            //
            // We kind of try to apply these rules:
            //
            // https://medium.com/@patrickbrosset/when-does-white-space-matter-in-html-b90e8a7cdd33

            var prevHadText = false
            var nodeBlock = isBlockElement(node)
            for (var sub = node.firstChild; sub; sub = sub.nextSibling) {
                var s

                if (sub.nodeType == Node.TEXT_NODE) {
                    // Text node and we cannot use innerText: start with raw value
                    s = sub.nodeValue || ''

                    // Two or more consecutive white space characters -> one space
                    s = s.replace(/[ \n\r\t]{2,}/g, ' ')

                    // Trim left
                    if (nodeBlock && sub == node.firstChild ||
                            sub.previousElementSibling && isBlockOrBreakElement(sub.previousElementSibling)) {
                        s = s.trimLeft()
                    }

                    // Trim right
                    if (nodeBlock && sub == node.lastChild ||
                            sub.nextElementSibling && isBlockOrBreakElement(sub.nextElementSibling)) {
                        s = s.trimRight()
                    }

                    prevHadText = s != ''
                } else {
                    // Element node (most likely)
                    s = ''
                    if (sub.nodeType == Node.ELEMENT_NODE && isBlockElement(sub) &&
                            (sub.previousElementSibling && !isBlockElement(sub.previousElementSibling) || prevHadText)) {
                        s += '\n'
                    }

                    s += collectSaveTextPlainImpl(sub, addQuotingPrefix, canUseInnerText)
                    if (needLineBreakAfterElementSubNode(sub)) {
                        s += '\n'
                    }

                    prevHadText = false
                }
                text += s
            }
        }

        if (nodeType == Node.ELEMENT_NODE && node.tagName == 'BLOCKQUOTE' &&
                nodeHasAttributeValueIgnoreCase(node, 'type', 'cite')) {
            text = text.trimRight();
            if (addQuotingPrefix) {
                text = text.replace(/^/mg, '>')
            }
            if (!text.endsWith('\n')) {
                text += '\n'
            }
        }
    } else if (nodeType == Node.TEXT_NODE || nodeType == Node.CDATA_SECTION_NODE) {
        text = node.nodeValue
    }

    return text
}

function collectSaveTextPlain(node, addQuotingPrefix) {
    return collectSaveTextPlainImpl(node, addQuotingPrefix, true)
}

function doSave(seed, isRichFormat, addQuotingPrefix) {
    consoleLog('doSave isRichFormat = ' + isRichFormat + ', addQuotingPrefix = ' + addQuotingPrefix)

    var styles = null
    var textHtml = null
    var originalHtml = null
    if (isRichFormat) {
        var listStyles = document.querySelectorAll('style')
        if (listStyles) {
            for (var i = listStyles.length - 1; i >= 0; -- i) {
                var nodeStyle = listStyles[i]
                if (nodeStyle.id != 'aqm-web-editor-styles') {
                    if (styles == null) {
                        styles = ''
                    } else {
                        styles += '\n'
                    }

                    styles += nodeStyle.outerHTML
                }
            }
        }

        textHtml = AQM_EDIT_AREA.innerHTML
        if (textHtml) {
            textHtml = textHtml.trim()

            // Check if the edit area has auto direction set
            for (var child = AQM_EDIT_AREA.firstChild; child; child = child.nextSibling) {
                if (isDivNode(child)) {
                    var attrDir = child.getAttribute('dir')
                    var attrId = child.getAttribute('id')

                    if (attrDir == 'auto' || attrId == 'aqm-signature' || attrId == 'aqm-greeting') {
                        continue
                    }
                } else if (isTextNode(child)) {
                    if (isTextNodeEmpty(child)) {
                        continue
                    }
                } else if (isCommentNode(child)) {
                    continue
                }

                // Found a node that needs it
                // TODO: maybe we can add divs or attributes in the right place instead of wrapping?
                textHtml = '<div dir="auto">\n' + textHtml + '\n</div>'
                break
            }
        }

        if (AQM_ORIGINAL) {
            originalHtml = AQM_ORIGINAL.outerHTML
        }
    }

    var textPlain = collectSaveTextPlain(AQM_EDIT_AREA, addQuotingPrefix)
    textPlain = textPlain.replace(/\u00A0/mg, ' ')
    textPlain = textPlain.replace(/^(>+)([^> \n])/mg, '$1 $2')

    // AQM-1651 - moved into Java side setSaveContent so it can preserve "-- " as special case
    // textPlain = textPlain.replace(/ +$/mg, '')

    consoleLogVerbose('Text plain: ' + textPlain)

    AQM_EDIT_CALLBACK.setSaveContent(seed, isRichFormat, styles, textHtml, originalHtml, textPlain)
}
