/*
 * Copyright (C) 2012 Google Inc.
 * Licensed to The Android Open Source Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var AQM_TRANSFORM_MINIMUM_EFFECTIVE_RATIO = 0.80;
var AQM_TRANSFORM_BETTER_ZOOM_THAN_FIT_1 = 1.20;
var AQM_TRANSFORM_BETTER_ZOOM_THAN_FIT_2 = 1.20;
var AQM_TRANSFORM_IMAGE_LOAD_LIST = [];
var AQM_CURRENT_VIEWPORT_WIDTH = 0;
var AQM_MIN_ZOOM = 0.2;

/*
 * Attr and class constants
 */

var AQM_CLASS_QUOTE_HIDDEN = "aqm-quote-hidden"
var AQM_ATTR_QUOTE_ID = "data-aqm-quote-id"

/*
 * Padding
 */
var AQM_PADDING_TWO = 2 * AQM_OVERLAY_PADDING;

/*
 * Are we still connected to Java code?
 */
var AQM_IS_CONNECTED = true;

function checkIsConnected() {
	if (!AQM_IS_CONNECTED) {
		return false;
	}

	AQM_IS_CONNECTED = aqm_bind.isConnected();
	if (!AQM_IS_CONNECTED) {
		consoleLog("checkIsConnected: *** Java side has gone away ***");
		return false;
	}

	return true;
}

function checkIsConnectedFast(i) {
	if (!AQM_IS_CONNECTED) {
		return false;
	}

	if (((i + 1) % 10) == 0 && !checkIsConnected()) {
		return false;
	}
	return true;
}

/*
 * New Chrome version?
 */
function isNewChrome(needVersion) {
    var userAgent = navigator.userAgent
    if (userAgent) {
        consoleLog("isNewChrome: " + userAgent)
        var i = userAgent.indexOf("Chrome/")
        if (i != -1) {
            var s = userAgent.substring(i + 7)
            var j = s.indexOf(".")
            if (j != -1) {
                s = s.substring(0, j)
                var version = parseInt(s)
                consoleLog("isNewChrome: " + version + ", need " + needVersion)
                return version >= needVersion
            }
        }
    }
    return false
}

var AQM_IS_NEW_CHROME_57 = AQM_API_VERSION >= 21 && isNewChrome(57)
var AQM_IS_NEW_CHROME_37 = AQM_API_VERSION >= 21 && isNewChrome(37)

/*
 * Measure and return overlay positions
 */
function measurePositions() {
	var itemTop = document.querySelector("#aqm-overlay-top");
	var itemBottom = document.querySelector("#aqm-overlay-bottom");
	var itemWrapper = document.querySelector("#aqm-message-wrapper");

	var offset = getTotalOffset(itemBottom);

	consoleLog("measurePositions: bottom offset left = " + offset.left + ", bottom offset left top = " + offset.top);

	aqm_bind.onWebContentGeometryChange(offset.left, offset.top);
}

/*
 * De-jitter content heights, to avoid infinite loops.
 */
function isContentSizeChanged(size0, size1) {
    return size0 > 0 && size1 > 0 &&
        (size0 > size1 * 105 / 100 || size1 > size0 * 105 / 100)
}

/*
 * Set overlay sizes
 */
function setOverlaySizes(sizeTop, sizeBottom) {
	var itemTop = document.querySelector("#aqm-overlay-top");
	var itemBottom = document.querySelector("#aqm-overlay-bottom");

	// Old scroll pos
	var scrollPos0 = window.scrollY
    var scrollDeltaY = 0

	consoleLog("setOverlaySizes: " + sizeTop + ", " + sizeBottom + ", scrollPos0 = " + scrollPos0);

    // Update overlay sizes, saving old values
    var sizeTop0 = 0, sizeTop1 = 0
	if (itemTop != null) {
	    sizeTop0 = itemTop.scrollHeight

		itemTop.style.height = sizeTop + "px"

		sizeTop1 = itemTop.scrollHeight
	    consoleLog("setOverlaySizes: top height " + sizeTop0 + " -> " + sizeTop1)

	    if (sizeTop0 != sizeTop1) {
	        callMeasure = true
	    }
	}

	var sizeBottom0 = 0, sizeBottom1 = 0
	if (itemBottom != null) {
	    sizeBottom0 = itemBottom.scrollHeight

		itemBottom.style.height = sizeBottom + "px";

	    sizeBottom1 = itemBottom.scrollHeight
	}

    // Adjust scroll position so the content doesn't jump
    if (scrollPos0 > 0 && sizeTop0 > 0 && sizeTop1 > 0) {
        if (AQM_IS_NEW_CHROME_57) {
            if (scrollPos0 > 0 && scrollPos0 < sizeTop0 && sizeTop1 > sizeTop0) {
                scrollDeltaY = sizeTop1 - sizeTop0
            } else if (scrollPos0 > 0 && scrollPos0 < sizeTop1 && sizeTop1 < sizeTop0) {
                scrollDeltaY = sizeTop1 - sizeTop0
            } else {
                scrollDeltaY = 0;
            }
        } else {
            scrollDeltaY = sizeTop1 - sizeTop0
        }
    }

    if (scrollDeltaY != 0) {
        window.scrollBy(0, scrollDeltaY)
        consoleLog("setOverlaySizes: did scrollBy " + scrollDeltaY)
    }

    // Report that we'd set overlay sizes back to Java
	aqm_bind.onWebOverlaySizesSet(sizeTop, sizeBottom);

    if (sizeTop0 != sizeTop1 || sizeBottom0 != sizeBottom1) {
        // Re-measure what we got
        measurePositions();
	}
}

/*
 * Called after changing font scale
 */
function onUserFontScaleChanged() {
	measurePositions();
}

/*
 * Returns the page offset of an element.
 * 
 * @param {Element} element The element to return the page offset for. @return {left: number, top: number} A tuple
 * including a left and top value representing the page offset of the element.
 */
function getTotalOffset(el) {
	var result = {
		left : 0,
		top : 0
	};
	var parent = el;

	while (parent) {
		result.left += parent.offsetLeft;
		result.top += parent.offsetTop;
		parent = parent.offsetParent;
	}

	return result;
}

/*
 * Hide quoted text
 */
function hideQuotedText() {
	var nodes, node, toggle;
	var i;
	var isHidden;
	var expandedMap = {}, expandedList;

	nodes = document.querySelectorAll(".aqm-quote");

	consoleLog("hideQuotedText: considering " + nodes.length + " hidden quote elements");

	if (nodes.length > 0) {
	    expandedList = aqm_bind.getExpandedQuoteIds()
        if (expandedList) {
	        var splits = expandedList.split(" ")
	        for (i = 0; i < splits.length; ++ i) {
	            var key = splits[i]
	            if (key) {
	                expandedMap[key] = true
	            }
	        }
	    }
	}

	for (i = 0; i < nodes.length; ++i) {
		node = nodes[i];
	    key = node.getAttribute(AQM_ATTR_QUOTE_ID);

		toggle = document.createElement("div");
		toggle.className = "aqm-quote-header";

		isHidden = node.classList.contains(AQM_CLASS_QUOTE_HIDDEN);
        if (isHidden && key) {
            if (expandedMap[key]) {
                node.classList.remove(AQM_CLASS_QUOTE_HIDDEN);
                isHidden = false
            }
        }
		toggle.innerHTML = isHidden ? AQM_QUOTE_SHOW_TEXT : AQM_QUOTE_HIDE_TEXT;

		toggle.setAttribute("dir", "auto");
		toggle.onclick = onQuotedToggleClick;
		node.parentNode.insertBefore(toggle, node);

		while (AQM_HIDE_QUOTED_MERGE && i + 1 < nodes.length) {
			var next = nodes[i + 1];
			var nodeParent = node.parentElement;
			var nextParent = next.parentElement;

			if (nodeParent != nextParent) {
				break;
			}
			consoleLog("hideQuotedText: node and next have same parent");

			var listMerge = [];
			var canMerge = true;
			for (var curr = node.nextSibling; curr && curr != next; curr = curr.nextSibling) {
				var currType = curr.nodeType;
				consoleLog("hideQuotedText: curr = " + curr + ", type = " + currType);

				if (currType == 8) {
					// Comments
					listMerge.push(curr);
					continue;
				}
				if (currType != 1 || curr.firstChild || curr.hasAttributes()) {
					// HTML nodes with no children or attributes
					canMerge = false;
					break;
				}
				var currName = curr.tagName;
				if (currName != "DIV") {
					// Only specific node types
					canMerge = false;
					break;
				}

				listMerge.push(curr);
				if (listMerge.length > 3) {
					// Too many nodes in between
					canMerge = false;
					break;
				}
			}

			if (!canMerge) {
				break;
			}
			
			var nextChildren = next.childNodes;
			if (nextChildren) {
				for (var j = 0; j < nextChildren.length; ++j) {
					listMerge.push(nextChildren[j]);
				}
			}

			consoleLog("hideQuotedText: can merge " + listMerge);
			for (var m = 0; m < listMerge.length; ++m) {
				node.appendChild(listMerge[m]);
			}

			nextParent.removeChild(next);
			++i;
		}
	}
}

function onQuotedToggleClick(e) {
	var toggle, node;
	var key;
	var isHidden;

	consoleLog("onQuotedToggleClick: " + e.target);

	toggle = e.target;
	node = toggle.nextSibling;
	key = node.getAttribute(AQM_ATTR_QUOTE_ID);

	isHidden = node.classList.contains(AQM_CLASS_QUOTE_HIDDEN);
	toggle.innerHTML = isHidden ? AQM_QUOTE_HIDE_TEXT : AQM_QUOTE_SHOW_TEXT;

	if (isHidden) {
	    // Expand
		node.classList.remove(AQM_CLASS_QUOTE_HIDDEN);
		aqm_bind.onQuoteExpanded(key);
		if (AQM_AUTO_FIT && checkIsConnected()) {
			performAutoFit();
		}
	} else {
	    // Collapse
		node.classList.add(AQM_CLASS_QUOTE_HIDDEN);
		aqm_bind.onQuoteCollapsed(key);
	}
}

/*
 * Top-level "shrink to fit"
 */
function performAutoFit() {
	var i;
	var origWidth;
	var oldZoom, newZoom;
	var el;
	var documentWidth = document.body.offsetWidth;
	var goalWidth = aqm_bind.getWebViewWidth() - AQM_PADDING_TWO;

	consoleLog("performAutoFit: documentWidth = " + documentWidth + ", goalWidth = " + goalWidth);

	if (goalWidth < 0) {
		consoleLog("performAutoFit: goalWidth < 0, ignoring");
		return;
	}

	var elements = document.querySelectorAll("#aqm-message-wrapper");

	for (i = 0; i < elements.length; ++i) {
		el = elements[i];
		oldZoom = el.style.zoom;
		if (oldZoom) {
			el.style.zoom = 1;
		}
		origWidth = el.style.width;
		el.style.width = goalWidth + "px";

		consoleLog("performAutoFit: el.scrollWidth = " + el.scrollWidth + ", origWidth = " + origWidth);

		if (goalWidth < el.scrollWidth) {
			transformContent(el, goalWidth, el.scrollWidth);
		}

		newZoom = goalWidth / el.scrollWidth;
		if (newZoom > 1) {
			newZoom = 1;
		} else if (newZoom < AQM_MIN_ZOOM) {
		    consoleLog("performAutoFit: pinning zoom " + newZoom + " to min " + AQM_MIN_ZOOM);
            newZoom = AQM_MIN_ZOOM;
		}
		el.style.zoom = newZoom;

		consoleLog("performAutoFit: el.scrollWidth = " + el.scrollWidth + ", newZoom = " + newZoom);

		el.style.width = origWidth;
	}
}

/*
 * Shrink to fit implementation
 */
function transformContent(el, docWidth, elWidth) {
	var start;
	var newWidth = elWidth;
	var beforeWidth;
	var nodes, touched;
	var done = false;
	var ratio;
	var actionLog = [];
	var tmpActionLog = [];
	var imageActionLog = [];

	consoleLog("transformContent: elWidth " + elWidth + ", docWidth = " + docWidth);

	if (elWidth <= docWidth) {
		return;
	}

    // Sometimes it's better to scale a little than to break markup
    ratio = newWidth / docWidth
    consoleLog("transform ratio, before anything = " + ratio)

    if (ratio <= AQM_TRANSFORM_BETTER_ZOOM_THAN_FIT_1) {
        consoleLog("transformContent without anything is effective enough");
        return;
    }

    // We need to work with markup
	start = Date.now();
	touched = false;

	// DIV and TEXTAREA elements
	if (!done && checkIsConnected()) {
		nodes = el.querySelectorAll("div[style], textarea[style]");
		consoleLog("transformContent: considering " + nodes.length + " div / textarea elements");
		touched = transformBlockElements(nodes, docWidth, actionLog);
		if (touched) {
			newWidth = el.scrollWidth;
			consoleLog("transformed div-width on el=" + el + ", oldW=" + elWidth + ", newW=" + newWidth + ", docW="
					+ docWidth);
			if (newWidth <= docWidth) {
				done = true;
			}
		}
	}

	// Images
	if (!done && checkIsConnected()) {
		nodes = el.querySelectorAll("img");
		consoleLog("transformContent: considering " + nodes.length + " image elements");
		touched = transformImages(nodes, docWidth, actionLog, imageActionLog);
		if (touched) {
			newWidth = el.scrollWidth;
			consoleLog("transformed images on el=" + el + ", oldW=" + elWidth + ", newW=" + newWidth + ", docW="
					+ docWidth);
			if (newWidth <= docWidth) {
				done = true;
			}
		}
	}

	// Anchors
	if (!done && checkIsConnected()) {
		nodes = el.querySelectorAll("a");
		consoleLog("transformContent: considering " + nodes.length + " anchor elements");
		touched = transforAnchors(nodes, docWidth, actionLog);
		if (touched) {
			newWidth = el.scrollWidth;
			consoleLog("transformed anchors on el=" + el + ", oldW=" + elWidth + ", newW=" + newWidth + ", docW="
					+ docWidth);
			if (newWidth <= docWidth) {
				done = true;
			}
		}
	}

	// Avoid breaking tables if we already achieved "good enough" scaling
	if (!done && newWidth > docWidth) {
        ratio = newWidth / docWidth
        consoleLog("transform ratio, before tables = " + ratio)

        if (ratio <= AQM_TRANSFORM_BETTER_ZOOM_THAN_FIT_2) {
            consoleLog("transformContent without tables is effective enough");

            // Undo transforms to images where we use actual WebView width - so they don't get scaled twice
            // by maxWidth and then by overall zoom factor
            if (imageActionLog && imageActionLog.length > 0) {
                consoleLog("transformContent: undoing old transforms to images");
                undoActions(imageActionLog)
            }

            done = true;
		}
	}

	// Tables
	if (!done && checkIsConnected()) {
		nodes = el.querySelectorAll("table");
		consoleLog("transformContent: considering " + nodes.length + " table elements");

		touched = addClassToElements(nodes, docWidth, shouldTransformTable, "aqm-transformed", actionLog);
		if (touched) {
			newWidth = el.scrollWidth;
            ratio = newWidth / docWidth
			consoleLog("transformed tables on el=" + el + ", oldW=" + elWidth + ", newW=" + newWidth + ", docW="
					+ docWidth + ", ratio = " + ratio);
			if (newWidth <= docWidth) {
				done = true;
			}
		}

        // Break words in tables
		if (!done && AQM_API_VERSION >= 19 && checkIsConnected()) {
		    var leaves = getLeafElements(nodes, "table")
		    if (leaves) {
                touched = addClassToElements(leaves, docWidth, shouldTransformTableWithWordBreak,
                    "aqm-word-break", actionLog);
                if (touched) {
                    newWidth = el.scrollWidth;
                    ratio = newWidth / docWidth
                    consoleLog("transformed tables on el=" + el + ", oldW=" + elWidth + ", newW=" + newWidth + ", docW="
                            + docWidth + ", ratio = " + ratio);
                    if (newWidth <= docWidth) {
                        done = true;
                    }
                }
            }
		}
	}

	// Table cells
	if (!done && checkIsConnected()) {
		beforeWidth = newWidth;
		nodes = el.querySelectorAll("td");
		consoleLog("transformContent: considering " + nodes.length + " td elements");
		touched = addClassToElements(nodes, docWidth, null /* all */, "aqm-transformed", tmpActionLog);
		if (touched) {
			newWidth = el.scrollWidth;
			consoleLog("transformed td on el=" + el + ", oldW=" + elWidth + ", newW=" + newWidth + ", docW=" + docWidth);
			if (newWidth <= docWidth) {
				done = true;
			} else if (newWidth >= beforeWidth) {
				// this transform did not improve things, and it is somewhat risky.
				// back it out, since it's the last transform and we gained nothing.
				consoleLog("transformed td did not help")
				undoActions(tmpActionLog);
			} else {
				// the transform WAS effective (although not 100%)
				// copy the temporary action log entries over as normal
				for (i = 0, len = tmpActionLog.length; i < len; i++) {
					actionLog.push(tmpActionLog[i]);
				}
			}
		}
	}

	// Was it good enough?
	ratio = newWidth / elWidth;
	consoleLog("transform ratio, final = " + ratio);
	if (!done && ratio <= AQM_TRANSFORM_MINIMUM_EFFECTIVE_RATIO) {
		consoleLog("transformContent deemed effective enough");

        // Undo transforms to images where we use actual WebView width - so they don't get scaled twice
        // by maxWidth and then by overall zoom factor
		if (imageActionLog && imageActionLog.length > 0) {
            consoleLog("transformContent: undoing old transforms to images");
            undoActions(imageActionLog)
        }

		done = true;
	}

	// Are we done?
	if (done) {
		consoleLog("transformContent succeeded, elapsed time=" + (Date.now() - start));
		return;
	}

	// No, couldn't do anything with this message
	undoActions(actionLog);
	if (actionLog.length > 0) {
		consoleLog("transformContent failed, changes reversed. elapsed time=" + (Date.now() - start));
	}
}

function shouldTransformTable(table, docWidth) {
	if (table.hasAttribute("width") || table.style.width) {
	    consoleLog("shouldTransformTable, width = " + table.scrollWidth)
	    return true
	}
	return false
}

function shouldTransformTableWithWordBreak(table, docWidth) {
    if (table.hasAttribute("width") || table.style.width) {
        var tableScrollWidth = table.scrollWidth
        if (tableScrollWidth >= docWidth * 2) {
            consoleLog("shouldTransformTableWithWordBreak, scrollWidth = " + tableScrollWidth)
        	return true
        }
    }

	return false
}

function getLeafElements(nodes, tagName) {
    var debug = AQM_DEBUG_LOG
    var tagNameUpper = tagName.toUpperCase()
    var roots = []
    var result = []
    var i, j, len, node, curr
    var list

    for (i = 0, len = nodes.length; i < len; i++) {
        node = curr = nodes[i]

        for (;;) {
            curr = curr.parentNode
            if (curr == null) {
                // Root node
                roots.push(node)
                break
            } else if (curr.nodeType == 1 && curr.tagName && curr.tagName.toUpperCase() == tagNameUpper) {
                // "node" is under "curr"
                list = curr.aqm_children_list
                if (!list) {
                    list = []
                    curr.aqm_children_list = list
                }
                list.push(node)
                break
            }
        }
    }

    if (debug) {
        // Debug
        for (i = 0, len = nodes.length; i < len; i++) {
            curr = nodes[i]
            consoleLog("Original " + i + " = " + curr + ", width = " + curr.scrollWidth)
        }

        // Debug
        consoleLog("Roots: " + roots);
        for (i = 0, len = roots.length; i < len; i++) {
            curr = roots[i]
            list = curr.aqm_children_list
            if (list) {
                consoleLog("Root " + i + " = " + curr + ", children = " + list)
            } else {
                consoleLog("Root " + i + " = " + curr + ", NO children")
            }
        }
    }

    // Extract leaf elements
    for (i = 0, len = nodes.length; i < len; i++) {
        curr = nodes[i]
        list = curr.aqm_children_list
        if (!list) {
            result.push(curr)
        }
    }

    if (debug) {
        // Debug
        for (i = 0, len = result.length; i < len; i++) {
            curr = result[i]
            consoleLog("Result " + i + " = " + curr + ", width = " + curr.scrollWidth)
        }
    }

    return result
}

function transformBlockElements(nodes, docWidth, actionLog) {
	var i, len;
	var node;
	var wStr;
	var index;
	var touched = false;

	for (i = 0, len = nodes.length; i < len; i++) {
		node = nodes[i];
		wStr = node.style.width || node.style.minWidth;
		index = wStr ? wStr.indexOf("px") : -1;
		if (index >= 0 && wStr.slice(0, index) > docWidth) {
			saveStyleProperty(node, "width", actionLog);
			saveStyleProperty(node, "minWidth", actionLog);
			saveStyleProperty(node, "maxWidth", actionLog);

			setStyleProperty(node, "width", "width", "100%");
			setStyleProperty(node, "minWidth", "min-width", "");
			setStyleProperty(node, "maxWidth", "max-width", wStr);

			node.setAttribute("data-aqm-touched", true);
			touched = true;
		}

		if (!checkIsConnectedFast(i)) {
			return false;
		}
	}
	return touched;
}

function transformImages(nodes, docWidth, actionLog, imageActionLog) {
	var i, len;
	var node;
	var w, h, height;
	var touched = false;

	for (i = 0, len = nodes.length; i < len; i++) {
		node = nodes[i];
		w = node.offsetWidth;
		h = node.offsetHeight;
		// shrink w/h proportionally if the img is wider than available width
		if (w > docWidth) {
			consoleLog("transformImages: w = " + w + ", docWidth = " + docWidth);

			var tempActionLog = [], tempIndex, tempLen, temp;

			saveStyleProperty(node, "maxWidth", tempActionLog);
			saveStyleProperty(node, "maxHeight", tempActionLog);
			saveStyleProperty(node, "width", tempActionLog);
			saveStyleProperty(node, "height", tempActionLog);

			for (tempIndex = 0, tempLen = tempActionLog.length; tempIndex < tempLen; tempIndex++) {
			    temp = tempActionLog[tempIndex]
			    actionLog.push(temp)
			    imageActionLog.push(temp)
			}

			height = node.style.height || node.getAttribute("height");

			setStyleProperty(node, "maxWidth", "max-width", docWidth + "px");
			// AQM-1073, AQM-1161: we're sticking with "100%" after all, and added menu -> view -> revert auto-fit
            setStyleProperty(node, "width", "width", "100%");
			if (height && isNumber(height)) {
				setStyleProperty(node, "maxHeight", "max-height", height + "px");
			}
			setStyleProperty(node, "height", "height", "auto");

			node.setAttribute("data-aqm-touched", true);
			touched = true;
		}

		if (!checkIsConnectedFast(i)) {
			return false;
		}
	}
	return touched;
}

function transforAnchors(nodes, docWidth, actionLog) {
	var i, len;
	var node;
	var w, h, height;
	var touched = false;

	for (i = 0, len = nodes.length; i < len; i++) {
		node = nodes[i];
		w = node.offsetWidth;
		
		if (w > docWidth) {
			consoleLog("transformAnchors: w = " + w + ", docWidth = " + docWidth);

			saveStyleProperty(node, "wordBreak", actionLog);
			setStyleProperty(node, "wordBreak", "word-break", "break-all");

			node.setAttribute("data-aqm-touched", true);
			touched = true;
		}

		if (!checkIsConnectedFast(i)) {
			return false;
		}
	}
	return touched;
}

function isNumber(val) {
	return !isNaN(val) && ('' + val).trim().length > 0;
}

function addClassToElements(nodes, docWidth, conditionFn, classToAdd, actionLog) {
	var i, len;
	var node;
	var added = false;
	for (i = 0, len = nodes.length; i < len; i++) {
		node = nodes[i];
		if (!conditionFn || conditionFn(node, docWidth)) {
			if (node.classList.contains(classToAdd)) {
				continue;
			}
			node.classList.add(classToAdd);
			added = true;
			actionLog.push([ node.classList.remove, node.classList, [ classToAdd ] ]);
		}
	}
	return added;
}

function saveElementWidth(node, actionLog) {
	var savedName = "data-aqm-original-node-" + "width";
	node.setAttribute(savedName, node.width);
	actionLog.push([ undoSetWidth, node, [ "width", savedName ] ]);
}

function undoSetWidth(property, savedProperty) {
	this.width = savedProperty ? this.getAttribute(savedProperty) : "";
}

function saveStyleProperty(node, property, actionLog) {
	var savedName = "data-aqm-original-style-" + property;
	node.setAttribute(savedName, node.style[property]);
	actionLog.push([ undoSetProperty, node, [ property, savedName ] ]);
}

function setStyleProperty(node, propertyIndex, propertyProp, value) {
	if (node.style.setProperty) {
		node.style.setProperty(propertyProp, value, "important");
	} else {
		node.style[propertyIndex] = value;
	}
}

function undoSetProperty(property, savedProperty) {
	this.style[property] = savedProperty ? this.getAttribute(savedProperty) : "";
}

function undoActions(actionLog) {
	var i, len;
	for (i = 0, len = actionLog.length; i < len; i++) {
	    var entry = actionLog[i];
        //consoleLog("undo " + i + ", entry = " + entry)

		entry[0].apply(entry[1], entry[2]);
	}
}

/*
 * Attach image loaders so we can auto-fit again when they're done loading
 */
function attachImageOnLoad() {
	var i, len;
	var originalSrc;
	var image;
	var images = document.querySelectorAll("#aqm-message-wrapper img");
	for (i = 0, len = images.length; i < len; i++) {
		image = images[i];
		originalSrc = image.src;
		if (!isEmptyString(originalSrc)) {
			image.src = '';
			image.onload = imageOnLoad;
			image.src = originalSrc;
		}
	}
}

function isEmptyString(s) {
	return s === undefined || s === null || s.length == 0;
}

function imageOnLoad(e) {
	var img = e.target;
	consoleLog("imageOnLoad: src = " + img.src);

	// if there was no previous work, schedule a new deferred job
	if (AQM_TRANSFORM_IMAGE_LOAD_LIST.length == 0) {
		consoleLog("imageOnLoad: enqueing imageOnAllLoaded");
		window.setTimeout(imageOnAllLoaded, 0);
	}

	// enqueue the work if it wasn't already enqueued
	if (AQM_TRANSFORM_IMAGE_LOAD_LIST.indexOf(img) == -1) {
		consoleLog("imageOnLoad: adding to list");
		AQM_TRANSFORM_IMAGE_LOAD_LIST.push(img);
	}
}

function imageOnAllLoaded() {
	consoleLog("imageOnAllLoaded: performing auto-fit again");
	if (checkIsConnected()) {
		performAutoFit();
		measurePositions();
	}
	AQM_TRANSFORM_IMAGE_LOAD_LIST = [];
}

/*
 * Disable forms
 */
function onSubmitPostForm() {
    alert(AQM_MSG_FORMS_ARE_DISABLED);
    return false;
}

function disablePostForms() {
	var forms = document.getElementsByTagName('FORM');
	var i, j, elements;

	for (i = 0; i < forms.length; ++i) {
		if (forms[i].method.toUpperCase() === 'POST') {
			forms[i].onsubmit = onSubmitPostForm;

			elements = forms[i].elements;
			for (j = 0; j < elements.length; ++j) {
				if (elements[j].type != 'submit') {
					elements[j].disabled = true;
				}
			}
		}
	}
}

/*
 * Preserve marked backgrounds (dark theme)
 */
function preserveMarkedBackgrounds() {
    var max = 20
    var count = 0
    var list = document.querySelectorAll('[data-aqm-preserve]')
    if (list && list.length) {
        consoleLog('Considering ' + (list.length) + ' root elements for data-aqm-preserve=true')

        for (var i = 0; i < list.length; ++ i) {
            // Collect (deep) all child element nodes of each found node
            var queue = []
            var head = 0

            var node = list[i]
            while (true) {
                for (var sub = node.firstElementChild; sub; sub = sub.nextElementSibling) {
                    queue.push(sub)
                }

                var len = queue.length
                if (head == len || len > max) {
                    break
                }
                node = queue[head]
                ++ head
            }

            // Make sure all nodes have the attribute we want. Limit how many child nodes we process,
            // just a "sanity limit" in case the original node is kind of like "the whole message".
            if (queue.length <= max) {
                while (queue.length > 0) {
                    var item = queue.pop()
                    if (!item.hasAttribute('data-aqm-preserve')) {
                        item.setAttribute('data-aqm-preserve', 'true')
                        ++ count
                    }
                }
            }
        }

        consoleLog('Converted ' + count + ' child elements to data-aqm-preserve=true')
    }
}

/*
 * DOM parser replacement for Android 4.*
 */
function parseDocumentPre37(text) {
    var doc = document.implementation.createHTMLDocument("")
    doc.documentElement.innerHTML = text
    return doc
}

/*
 * Set text
 */

function setContentText(seed, sizeTop, sizeBottom, optionAutoFit, optionPreserveBackgrounds) {
    consoleLog('setContentText seed = ' + seed +
        ', sizeTop = ' + sizeTop +
        ', sizeBottom = ' + sizeBottom +
        ', optionAutoFit = ' + optionAutoFit +
        ', optionPreserveBackgrounds = ' + optionPreserveBackgrounds)

    var text = aqm_bind.getContentText(seed)
    if (text) {
        var fragment

        // Parse text into DOM tree
        if (AQM_IS_NEW_CHROME_37) {
            var parser = new DOMParser()
            fragment = parser.parseFromString(text, 'text/html')
        } else {
            fragment = parseDocumentPre37(text)
        }

        if (fragment && fragment.body) {
        	var itemTop = document.querySelector("#aqm-overlay-top");
        	var itemBottom = document.querySelector("#aqm-overlay-bottom");

            // Update overlay sizes
        	if (itemTop != null) {
        		itemTop.style.height = sizeTop + "px"
        	}
        	if (itemBottom != null) {
        		itemBottom.style.height = sizeBottom + "px";
        	}

            // Update content text
            var currOuter = document.getElementById('aqm-message-outer')
            var textOuter = fragment.getElementById('aqm-message-outer')

            if (currOuter && textOuter) {
                consoleLog('currOuter: ' + currOuter + ', textOuter: ' + textOuter + ", switching!")

                if (/* AQM_IS_NEW_CHROME_37 && */ currOuter.parentElement) {
                    // We should end up here
                    currOuter.parentElement.replaceChild(textOuter, currOuter)
                } else {
                    // ... but just in case
                    while (currOuter.lastChild) {
                        currOuter.removeChild(currOuter.lastChild)
                    }
                    while (textOuter.firstChild) {
                        currOuter.appendChild(textOuter.firstChild)
                    }
                }

                // Massage the content
                if (AQM_HIDE_QUOTED) {
                    hideQuotedText()
                }

                AQM_AUTO_FIT = optionAutoFit
                if (AQM_AUTO_FIT) {
                    attachImageOnLoad()
                    performAutoFit()
                }

                AQM_PRESERVE_BACKGROUNDS = optionPreserveBackgrounds
                if (AQM_PRESERVE_BACKGROUNDS && AQM_IS_NEW_CHROME_37) {
                    preserveMarkedBackgrounds()
                }

                disablePostForms()
                measurePositions()
            }
        } else {
            consoleLog('Could not parse text ' + fragment)
        }
    } else {
        consoleLog('Could not get text')
    }
}

/*
 * Logging
 */
function consoleLog(s) {
//	if (AQM_DEBUG_LOG) {
		console.log(s);
//	}
}

/*
 * Init
 */

aqm_bind.onInitDone();

